/**
 * @file * This is ModLocalFileSystem sample program.
 *
 * <pre>
 * 1. Compile this program and write to your mbed.
 * 2. Write files for testing to mbed disk.
 * 2. Access via web browser to http://192.168.0.39/local/<filename>
 * 3. You can see <filename> file on browser.
 * </pre>
 */
#include "mbed.h"
#include "rtos.h"
#include "SDFileSystem.h"
#include "mimic.h"
#include "utils/PlatformInfo.h"
#include "fsdata.h"

DigitalOut mbedled(LED1);
DigitalOut lpcxled(P0_22);

/**
 * local filesystem support.
 */
LocalFileSystem2 lf("local");
SDFileSystem sd(p5, p6, p7, p8,"sd");

unsigned int p;
/**
 * MiMic RemoteMCU httpd.<br/>
 * Number of simultaneous connections:4
 * <p>Service list</p>
 * <pre>
 * /local/ - mbed LocalFileSystem
 * </pre>
 */
class FsHttpd:public MiMic::Httpd
{
private:
    ModLocalFileSystem modlocal;
    ModLocalFileSystem modsd;
    ModUPnPDevice modupnp;
    ModRomFiles modromfs; //ROM file module
    
public:
    FsHttpd(Net& i_net,NetConfig& i_cfg):Httpd(i_cfg.getHttpPort())
    {
    
        this->modromfs.setParam("rom",FSDATA,1);
        //bind local file system path to /local/*
        modlocal.setParam("local");
        modsd.setParam("sd",ModLocalFileSystem::FST_SDFATFS);
        this->modupnp.setParam(i_net);
    }
    virtual void onRequest(HttpdConnection& i_connection)
    {
        p++;
        switch(PlatformInfo::getPlatformType()){
        case PlatformInfo::PF_MBED:
            mbedled = p%2;
            break;
        case PlatformInfo::PF_LPCXPRESSO:        
            lpcxled = p%2;
            break;
        }
        //try to ModRomFS module.
        if(this->modromfs.execute(i_connection)){
            return;
        }        
        //try to ModLocalFileSystem
        if(this->modlocal.execute(i_connection)){
            return;
        }
        //try to ModLocalFileSystem(SD)
        if(this->modsd.execute(i_connection)){
            return;
        }
        if(this->modupnp.execute(i_connection)){
            return;
        }
        //Otherwise, Send simple top index page.
        i_connection.sendHeader(200,"text/html",NULL);
        if(i_connection.isMethodType(Http::MT_GET)){
            i_connection.sendBodyF(
                "<!DOCTYPE html>"
                "<html lang=\"ja\">"  
                "<head></head>"
                "<body>"
                "<h1>This is MiMic Server!</h1>"
                "<hr/>"
                "<ul>"
                "<li><a href=\"/local/\">mbed Local Filesystem</a></li>"
                "<li><a href=\"/sd/\">SDCard</a></li>"
                "</ul></body>");
        }
    }
};

FsHttpd* httpd; //create a httpd instance.

int main()
{
    NetConfig cfg; //create network configulation
    MiMicNetIf netif;
    Net net(netif);//Net constructor must be created after started RTOS
    //Prepare configulation.
    cfg.setUPnPIcon(64,64,8,"image/png","/rom/icon.png");
    cfg.setUPnPUdn(0xe29f7102,0x4ba2,0x01e0,0);
    cfg.setFriendlyName("MbedFileServer");

    //try to override setting by local file.
    if(!cfg.loadFromFile("/local/mimic.cfg")){
        Thread::wait(2000);//wait for SD card initialization.
        cfg.loadFromFile("/sd/mimic.cfg");
    }
    httpd=new FsHttpd(net,cfg); //create a httpd instance.
    net.start(cfg);
    httpd->loop();  //start httpd loop.
    return 0;
}
