/** A sample code for "mini board PCU9669/PCA9665"
 *
 *  @author  Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version 1.1
 *  @date    11-Jul-2012
 *
 *  Released under the MIT License: http://mbed.org/license/mit
 *
 *  An operation sample of PCU9669/PCA9665 I2C bus controller.
 *  The mbed accesses the bus controller's parallel port (8/2 bit address and 8 bit data) by bit-banging.
 *  The bit-banging is poerformed by PortInOut function of mbed library.
 *
 *    To make the code porting easier, all codes are partitioned into layers to abstract other parts.
 *    The mbed specific parts are concentrated in lowest layer: "hardware_abs.*".
 *    This module may need to be modified for the porting.
 *
 *    All other upper layers are writen in standard-C.
 *
 *    base code is written from 05-Sep-2011 to 09-Sep-2011.
 *    And demo code has been build on 11-Sep-2011.
 *    Debug and code adjustment has been done on 08-Sep-2011.
 *    Small sanitization for main.cpp. All mbed related codes are moved in to "hardware_abs.*". 13-Oct-2011
 *    hardware_abs are moved into parallel_bus library folder, 3 LED driver operation sample 13-Feb.-2012
 *    PCU9669 and PCA9665 codes are packed in a project 14-Feb-2012. 
 *    
 *    Before builidng the code, please edit the file mini_board_PCU9669/config.h
 *    Un-comment the target name what you want to target. 
 */

/** Hardware abstraction layer module
 *
 *  All MCU hardware related code are defined in this module
 *  This module may need to be modified when the code is ported to other MCU. 
 */

#ifndef MINIBOARD_HARDWARE_ABS__
#define MINIBOARD_HARDWARE_ABS__

/** @def BURST_DATA_ACCESS
 *
 *  To accelerate multiple bus access on same addess, use BURST_DATA_ACCESS
 *  On the mbed emvironment, this burst access enables 3 times faster read/write compare to single access.
 *  For code porting, the BURST_DATA_ACCESS code part is not neccesary if the hardware is fast enough. 
 */
#define     BURST_DATA_ACCESS


/** @def ASSERT / DEASSERT
 *
 *  Reset signal logic difinition 
 */
#define     ASSERT      0   //  for hardware reset
#define     DEASSERT    1   //  for hardware reset


/** Install an ISR
 *
 *  Registering function as ISR. 
 *  The function will be called when the interrupt asserted.
 *
 *  @param fptr a pointer to a function
 */
void    install_ISR( void (*fptr)(void) );

/** Hardware initialization
 *
 *  MCU side initialization should be done in this function. 
 *  For the mbed, it set the initial state of parallel bus control signal. 
 */
void    hardware_initialize( void );

/** Reset signal control
 *
 *  This function drives the RESET signal line with given state.  
 *
 *  @param signal the state of RESET signal: ASSERT | DEASSERT
 */
void    hardware_reset( char signal );

/** Hardware reset
 *
 *  Asserts the RESET signal with required pulse width and waits its recovery time
 *  
 *  @param reset_pulse_width_us RESET pulse width in micro-seconds
 *  @param reset_recovery_us RESET  recovery time in micro-seconds (wait time after RESET de-assertion)
 */
void reset( int reset_pulse_width_us, int reset_recovery_us );

/** Triger signal control
 *
 *  This function drives the TRIGGER signal line with given state.  
 *
 *  @param signal the state of TRIGGER signal: ASSERT | DEASSERT
 */
void    hardware_trigger( char signal );

/** Write data
 *
 *  Writes 1 byte  
 *
 *  @param addr 8 bit address where the data should be written to 
 *  @param data 8 bit data which will be written
 *  @see read_data()
 */
void    write_data( char addr, char data );

/** Read data
 *
 *  Reads 1 byte  
 *
 *  @return 8 bit data which is read from bus
 *  @param addr 8 bit address where the data should be read from 
 *  @see write_data()
 */
char    read_data( char addr );

/** Wait micro-seconds
 *
 *  Wait function waits given-micro-seconds
 *
 *  @param micro-seconds the program should wait (32 bit integer value)
 *  @see wait_sec()
 */
void    hw_wait_us( int v );

/** Wait seconds
 *
 *  Wait function waits given-seconds
 *
 *  @param seconds the program should wait (float value)
 *  @see hw_wait_us()
 */
void    wait_sec( float );


/** BURST_DATA_ACCESS option code
 *
 *  This code is option to accelerate the bus access
 */
#ifdef  BURST_DATA_ACCESS

/** Write data
 *
 *  Writes multiple bytes to same address. 
 *  This function suitable to use for the registers like SLATABLE, TRANCONFIG and data buffers (through DATA register).
 *  While this access is going, the interrupt will be tempolary disabled (ISR execution will be postponed)
 *
 *  @param addr   8 bit address where the data should be written to 
 *  @param *data  pointer to char array. The data in this array will be written
 *  @param length length of the data (bytes)
 *  @see read_data_burst()
 */
void    write_data_burst( char addr, char *data, char length );

/** Read data
 *
 *  Reads multiple bytes from same address. 
 *  This function suitable to use for the registers like SLATABLE, TRANCONFIG and data buffers (through DATA register).
 *  While this access is going, the interrupt will be tempolary disabled (ISR execution will be postponed)
 *
 *  @param addr   8 bit address where the data should be written to 
 *  @param *data  pointer to char array. The read data will be written to this
 *  @param length length of the data (bytes)
 *  @see write_data_burst()
 */
void    read_data_burst( char addr, char *data, char length );
#endif

#endif  //   MINIBOARD_HARDWARE_ABS__

