/** A sample code for "mini board PCU9669/PCA9665"
 *
 *  @author  Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version 1.0
 *  @date    26-Mar-2012
 * 
 *  Released under the MIT License: http://mbed.org/license/mit
 *
 *  An operation sample of PCU9669/PCA9665 I2C bus controller.
 *  The mbed accesses the bus controller's parallel port (8/2 bit address and 8 bit data) by bit-banging.
 *  The bit-banging is poerformed by PortInOut function of mbed library.
 *
 *    To make the code porting easier, all codes are partitioned into layers to abstract other parts.
 *    The mbed specific parts are concentrated in lowest layer: "hardware_abs.*".
 *    This module may need to be modified for the porting.
 *
 *    All other upper layers are writen in standard-C.
 *
 *    base code is written from 05-Sep-2011 to 09-Sep-2011.
 *    And demo code has been build on 11-Sep-2011.
 *    Debug and code adjustment has been done on 08-Sep-2011.
 *    Small sanitization for main.cpp. All mbed related codes are moved in to "hardware_abs.*". 13-Oct-2011
 *    hardware_abs are moved into parallel_bus library folder, 3 LED driver operation sample 13-Feb.-2012
 *    PCU9669 and PCA9665 codes are packed in a project 14-Feb-2012.
 *
 *    Before builidng the code, please edit the file mini_board_PCU9669/config.h
 *    Un-comment the target name what you want to target.
 */

/** @note Application layer module of PCU9669
 *
 *  This code is made to explain basic PCU9669 operation.
 *  And also, it demonstrates 3 channels of I2C operation with several slaves on those buses.
 *
 *  The ch0 is an Fm+ bus that has a PCA9955 as slave device
 *  The ch1 and ch2 are UFm buses. Each bus has a PCU9955 as slave devices.
 */

#include "config.h"
#include "hardware_abs.h"
#include "transfer_manager.h"   //  abstracting the access of PCU9669 internal buffer
#include "PCU9669_access.h"     //  PCU9669 chip access interface
#include "PCx9955_reg.h"        //  slave specific definitions
#include "PCA9629_reg.h"
#include "utility.h"            //  
//#include "mbed.h"   //  this header is required only when printf() is used.

#if defined( CODE_FOR_PCU9669 ) || defined( CODE_FOR_PCA9663 )

#ifdef  CODE_FOR_PCU9669
#define     TARGET_CHIP_ID  PCU9669_ID
#endif

#ifdef  CODE_FOR_PCA9663
#define     TARGET_CHIP_ID  PCA9663_ID
#endif

#define RESET_PULSE_WIDTH_US    10  //  Minimum pulse width is 4us for PCU9669
#define RESET_RECOVERY_US       1000

/**
 *  register settings for PCx9955
 */

char  PCx9955_reg_data[] = {
    0x80,                                       //  Strat register address with AutoIncrement bit
    0x00, 0x05,                                 //  MODE1, MODE2
    0xAA, 0xAA, 0xAA, 0xAA,                     //  LEDOUT[3:0]
    0x80, 0x00,                                 //  GRPPWM, GRPFREQ
    PWM_INIT,  PWM_INIT,  PWM_INIT,  PWM_INIT,  //  PWM[3:0]
    PWM_INIT,  PWM_INIT,  PWM_INIT,  PWM_INIT,  //  PWM[7:4]
    PWM_INIT,  PWM_INIT,  PWM_INIT,  PWM_INIT,  //  PWM[11:8]
    PWM_INIT,  PWM_INIT,  PWM_INIT,  PWM_INIT,  //  PWM[15:12]
    IREF_INIT, IREF_INIT, IREF_INIT, IREF_INIT, //  IREF[3:0]
    IREF_INIT, IREF_INIT, IREF_INIT, IREF_INIT, //  IREF[7:4]
    IREF_INIT, IREF_INIT, IREF_INIT, IREF_INIT, //  IREF[11:8]
    IREF_INIT, IREF_INIT, IREF_INIT, IREF_INIT, //  IREF[15:12]
    0x08                                        //  OFFSET: 1uS offsets
};


char  PCx9955_reg_read_start_address    = 0x80;

char  read_buffer[ 41 ];

transaction     ufm_transactions[]   =   {
    { PCx9955_ADDR0, PCx9955_reg_data, sizeof( PCx9955_reg_data ) }
};

transaction     fm_plus_transactions[]   =   {
    { PCx9955_ADDR0, PCx9955_reg_data, sizeof( PCx9955_reg_data ) },
    { PCx9955_ADDR0, &PCx9955_reg_read_start_address, 1 },
    { PCx9955_ADDR0 | 0x01, read_buffer, sizeof( read_buffer ) }
};

char    read_done   = 0;

void interrupt_handler( void );

int main() {
    char    clear[ 16 ] = { 0 };
    int     count       = 0;
    int     i;
    int     j;

//    printf( "\r\nPCU9669 simple demo program on mbed\r\n  build : %s (UTC), %s \r\n\r\n", __TIME__, __DATE__ );

    hardware_initialize();          //  initializing bit-banging parallel port
    reset( RESET_PULSE_WIDTH_US, RESET_RECOVERY_US );  //  assert hardware /RESET sgnal

    if ( start_bus_controller( TARGET_CHIP_ID ) )   //  wait the bus controller ready and check chip ID
        return 1;

    write_ch_register( CH_FM_PLUS, INTMSK, 0x30 );  //  set bus controller to ignore NAK return from Fm+ slaves
    install_ISR( &interrupt_handler );              //  interrupt service routine install

    setup_transfer( CH_FM_PLUS, fm_plus_transactions, sizeof( fm_plus_transactions ) / sizeof( transaction ) );
    setup_transfer( CH_UFM1, ufm_transactions, sizeof( ufm_transactions ) / sizeof( transaction ) );
    setup_transfer( CH_UFM2, ufm_transactions, sizeof( ufm_transactions ) / sizeof( transaction ) );

    while ( 1 ) {
        for ( i = 0; i < 16; i++ ) {
            for ( j = 0; j < 256; j += 4 ) {
                buffer_overwrite( CH_FM_PLUS, 0, 9 + i, (char *)&j, 1 );
                buffer_overwrite( CH_UFM1,    0, 9 + i, (char *)&j, 1 );
                buffer_overwrite( CH_UFM2,    0, 9 + i, (char *)&j, 1 );

                read_done   = 0;

                set_n_of_transaction( CH_FM_PLUS, (sizeof( fm_plus_transactions ) / sizeof( transaction )) - ((count % 256) ? 1 : 0) );

                start( CH_FM_PLUS );
                start( CH_UFM1 );
                start( CH_UFM2 );

                wait_sec( 0.01 );
                count++;
#if 0
                if ( read_done ) {
                    buffer_read( CH_FM_PLUS, 2, 0, read_buffer, sizeof( read_buffer ) );
                    dump_read_data( read_buffer, sizeof( read_buffer ) );
                    read_done   = 0;
                }
#endif
            }
        }
        buffer_overwrite( CH_FM_PLUS, 0, 9, clear, 16 );
        buffer_overwrite( CH_UFM1,    0, 9, clear, 16 );
        buffer_overwrite( CH_UFM2,    0, 9, clear, 16 );
    }
}

void interrupt_handler( void ) {
    char    global_status;
    char    channel_status;

    global_status  = read_data( CTRLSTATUS );

    if ( global_status & 0x01 ) {  //  ch0
        channel_status  = read_ch_register( 0, CHSTATUS );

        if ( channel_status & 0x80 )
            read_done   = 1;
    }
    if ( global_status & 0x02 ) {
        channel_status  = read_ch_register( 1, CHSTATUS );
    }
    if ( global_status & 0x04 ) {
        channel_status  = read_ch_register( 2, CHSTATUS );
    }
//    printf( "ISR channel_status 0x%02X\r\n", channel_status );
}

#endif  //  CODE_FOR_PCU9669






