/*
 *  A sample code for PCF2127 library
 *
 *  @author     Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version    1.8
 *  @date       10-Dec-2014
 *
 *  PCF2127 is a "real time clock (RTC)" module which is including a Xtal and TCXO
 *  http://www.nxp.com/products/interface_and_connectivity/real_time_clocks/rtcs_with_temp_compensation/series/PCF2127.html
 *
 *  RTC initializing part is ported from..
 *    http://mbed.org/users/roen/notebook/real-time/
 *
 *  This code is refined version of..
 *    http://developer.mbed.org/users/okano/code/NXP_PCF2127A/
 */

#include "mbed.h"
#include "PCF2127.h"

PCF2127 rtc( p28, p27 );

void    show_time( void );
void    set_time( void );

void    RAM_dump( void );       //  for PCF2127 only
void    RAM_read_test( void );  //  for PCF2127 only
void    RAM_write_test( void ); //  for PCF2127 only
void    RAM_test( int last_RAM_content_invalid );       //  for PCF2127 only

int main()
{
    printf( "PCF2127 demo started.\r\n" );

    int clock_integrity_fail    = rtc.is_init_required();

    if ( clock_integrity_fail ) {
        set_time();
    }

    /*
     *  Next line can be enabled when using PCF2127 only
     */
    //RAM_test( clock_integrity_fail );

    while ( 1 ) {
        show_time();
        wait( 0.25 );
    }
}

void show_time( void )
{
    struct tm   dt, *dtp;
    time_t      t;
    char        s[ 30 ];

    dtp     = &dt;

    rtc.clear_intr();

    //  get time information from PCF2127
    if ( PCF2127::TIME_FUNC_ERROR == (t   = rtc.time( NULL )) ) {
        printf( "error at reading time\r\n" );
        return;
    }

    dtp     = localtime( &t );

    //  print time and date on terminal
    strftime( s, 30, "%H:%M:%S, %Y/%b/%d %a", dtp );
    printf( "%s\r\n", s );
}

void set_time( void )
{
#define MAX_CHAR_LENGTH 21
    char    s[ MAX_CHAR_LENGTH ];

    printf( "Enter current date and time:\r\n" );
    printf( "YYYY MM DD HH MM SS[enter]\r\n" );

    fgets( s, MAX_CHAR_LENGTH, stdin );
    printf( "user input: \"%s\"\r\n", s );

    rtc.set_time( s );
}

void RAM_function_test( void )
{
}

void RAM_dump( void )
{
    char    b[ 512 ];

    if ( rtc.RAM_read( 0, b, 512 ) ) {
        printf( "error at reading RAM\r\n" );
    }

    for ( int i = 0; i < 512; i++ ) {
        if ( !(i % 16) ) {
            printf( "\r\n  0x%03X(%03d): 0x", i, i );
        }
        printf( " %02X", b[ i ] );
    }
}

const char *texts[] = {
    "Love the life you live. Live the life you love.",
    "It always seems impossible until it's done.",
    "You'll never find a rainbow if you're looking down.",
    "There is always light behind the clouds."
};

void RAM_test( int last_RAM_content_invalid )
{
    char    txt[ 64 ];
    time_t  t;

    if ( !last_RAM_content_invalid ) {
        rtc.RAM_read( 0, (char *)&t, sizeof( t ) );
        rtc.RAM_read( 16, txt, sizeof( txt ) );
        printf( "RAM test read\r\n" );
        printf( "  last program start was on %s\r", ctime( &t ) );
        printf( "  last saved text was \"%s\"\r\n", txt );
    }

    t   = rtc.time( NULL );
    rtc.RAM_write( 0, (char *)&t, sizeof( t ) );
    rtc.RAM_write( 16, (char *)texts[ t % 4 ], strlen( texts[ t % 4 ] ) + 1 );
}

