/*
 *  PCF2127 library
 *
 *  @author     Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version    1.8
 *  @date       10-Dec-2014
 *
 *  PCF2127 is a "real time clock (RTC)" module which is including a Xtal and TCXO
 *  http://www.nxp.com/products/interface_and_connectivity/real_time_clocks/rtcs_with_temp_compensation/series/PCF2127.html
 *
 *  RTC initializing part is ported from..
 *    http://mbed.org/users/roen/notebook/real-time/
 *
 *  This code is refined version of..
 *    http://developer.mbed.org/users/okano/code/NXP_PCF2127A/
 */

#include    "mbed.h"
#include    "PCF2127.h"

#define     PCF2127_I2C_SLAVE_ADDRESS   0xA2

PCF2127::PCF2127( PinName sda, PinName sdl, char vControl_1, char vControl_2, char vControl_3, char vCLKOUT_ctl )
    : i2c_p( new I2C( sda, sdl ) ), i2c( *i2c_p ), device_address( PCF2127_I2C_SLAVE_ADDRESS )
{
    init( vControl_1, vControl_2, vControl_3, vCLKOUT_ctl );
}

PCF2127::PCF2127( I2C &i2c_, char vControl_1, char vControl_2, char vControl_3, char vCLKOUT_ctl )
    : i2c_p( NULL ), i2c( i2c_ ), device_address( PCF2127_I2C_SLAVE_ADDRESS )
{
    init( vControl_1, vControl_2, vControl_3, vCLKOUT_ctl );
}

PCF2127::~PCF2127()
{
    if ( NULL != i2c_p )
        delete  i2c_p;
}

int PCF2127::init( char vControl_1, char vControl_2, char vControl_3, char vCLKOUT_ctl )
{
    char    data[ 4 ];
    int     err;

    data[ 0 ]   = Control_1; //  access start register address
    data[ 1 ]   = vControl_1;
    data[ 2 ]   = vControl_2;
    data[ 3 ]   = vControl_3;

    err     = i2c.write( device_address, data, sizeof( data ) );
    err    |= set_register( CLKOUT_ctl, vCLKOUT_ctl );

    return ( err ? I2C_ACCESS_FAIL : NO_ERROR );
}

int PCF2127::is_init_required( void )
{
    return ( read_register( Seconds ) & 0x80 ? true : false );
}

int PCF2127::set_time( struct tm *dtp )
{
    char        buf[ 8 ];
    char        err;

    buf[ 0 ]    = Seconds;
    buf[ 1 ]    = i2bcd( dtp->tm_sec  );
    buf[ 2 ]    = i2bcd( dtp->tm_min  );
    buf[ 3 ]    = i2bcd( dtp->tm_hour );
    buf[ 4 ]    = i2bcd( dtp->tm_mday );
    buf[ 5 ]    = i2bcd( dtp->tm_wday );
    buf[ 6 ]    = i2bcd( dtp->tm_mon  + 1   );
    buf[ 7 ]    = i2bcd( dtp->tm_year - 100 );

    err = i2c.write( device_address, buf, 8 );

    return ( err ? I2C_ACCESS_FAIL : NO_ERROR );
}

int PCF2127::set_time( time_t *tp )
{
    return ( set_time( localtime( tp ) ) );
}

int PCF2127::set_time( char *s )
{
    //  The time information should be given in format of "YYYY MM DD HH MM SS"

    struct tm   dt, *dtp;

    dtp = &dt;

    sscanf( s, "%d %d %d %d %d %d", &(dtp->tm_year), &(dtp->tm_mon), &(dtp->tm_mday), &(dtp->tm_hour), &(dtp->tm_min), &(dtp->tm_sec) );
    printf( "%02d/%02d/%02d - %02d:%02d:%02d\r\n", (dtp->tm_year), (dtp->tm_mon), (dtp->tm_mday), (dtp->tm_hour), (dtp->tm_min), (dtp->tm_sec) );

    // adjust for tm structure required values
    dtp->tm_year = dtp->tm_year - 1900;
    dtp->tm_mon  = dtp->tm_mon - 1;

    return ( set_time( dtp ) );
}

time_t PCF2127::time( time_t *tp )
{
    struct tm   dt, *dtp;
    time_t      t;
    char        buf[ 8 ];

    dtp = &dt;

    buf[ 0 ]    = Seconds;  //  read start register address

    if ( i2c.write( device_address, buf, 1 ) )
        return ( TIME_FUNC_ERROR );

    if ( i2c.read( device_address, buf, 7 ) )
        return ( TIME_FUNC_ERROR );

    dtp->tm_sec     = bcd2i( buf[ 0 ] );
    dtp->tm_min     = bcd2i( buf[ 1 ] );
    dtp->tm_hour    = bcd2i( buf[ 2 ] );
    dtp->tm_mday    = bcd2i( buf[ 3 ] );
    dtp->tm_wday    = bcd2i( buf[ 4 ] );
    dtp->tm_mon     = bcd2i( buf[ 5 ] ) - 1;
    dtp->tm_year    = bcd2i( buf[ 6 ] ) + 100;

    t   = mktime( dtp );

    if ( tp )
        *tp  = t;

    return( t );
}

int PCF2127::set_alarm( char addr, char s )
{
    char    v;

    v   = i2bcd( s );

    return ( set_register( addr, v ) );
}

int PCF2127::clear_intr( void )
{
    return ( set_register( Control_2, 0x00 ) );
}

int PCF2127::RAM_write( int address, char *p, int size )
{
    char    b[ size + 1 ];

    b[ 0 ]  = RAM_wrt_cmd;

    for ( int i = 1; i <= size; i++ )
        b[ i ]  = *p++;

    if ( set_RAM_address( address ) )
        return ( I2C_ACCESS_FAIL );

    return ( i2c.write( device_address, b, sizeof( b ) ) ? I2C_ACCESS_FAIL : NO_ERROR );
}

int PCF2127::RAM_read( int address, char *p, int size )
{
    char    b   = RAM_rd_cmd;

    if ( set_RAM_address( address ) )
        return ( I2C_ACCESS_FAIL );

    if ( i2c.write( device_address, &b, 1 ) )
        return ( I2C_ACCESS_FAIL );

    return ( i2c.read( device_address, p, size ) ? I2C_ACCESS_FAIL : NO_ERROR );
}


int PCF2127::set_RAM_address( char address )
{
    char    b[ 3 ];

    b[ 0 ]  = RAM_addr_MSB;
    b[ 1 ]  = (address >> 8) & 0x1;
    b[ 2 ]  = address & 0xFF;

    return ( i2c.write( device_address, b, sizeof( b ) ) );
}

int PCF2127::set_register( char addr, char data )
{
    char    b[ 2 ];

    b[ 0 ]    = addr;
    b[ 1 ]    = data;

    return ( i2c.write( device_address, b, sizeof( b ) ) );
}

int PCF2127::read_register( char addr )
{
    char    data;

    data    = addr;
    i2c.write( device_address, &data, 1 );
    i2c.read( device_address, &data, 1 );

    return ( data );
}

char PCF2127::i2bcd( char n )
{
    return ( ((n / 10) << 4) | (n % 10) );
}

char PCF2127::bcd2i( char bcd )
{
    return ( ((bcd >> 4) * 10) + (bcd & 0x0F) );
}
