/*
 *  PCF2127 library
 *
 *  @author     Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version    1.7
 *  @date       10-Dec-2014
 *
 *  PCF2127 is a "real time clock (RTC)" module which is including a Xtal and TCXO
 *  http://www.nxp.com/products/interface_and_connectivity/real_time_clocks/rtcs_with_temp_compensation/series/PCF2127.html
 *
 *  RTC initializing part is ported from..
 *    http://mbed.org/users/roen/notebook/real-time/
 *
 *  This code is refined version of..
 *    http://developer.mbed.org/users/okano/code/NXP_PCF2127A/
 */

#ifndef        MBED_PCF2127
#define        MBED_PCF2127

#include    "mbed.h"

class PCF2127
{
public:

    /** name of the PCF2127 registers */
    typedef enum {
        Control_1,
        Control_2,
        Control_3,
        Seconds,
        Minutes,
        Hours,
        Days,
        Weekdays,
        Months,
        Years,
        Second_alarm,
        Minute_alarm,
        Hour_alarm,
        Day_alarm,
        Weekday_alarm,
        CLKOUT_ctl,
        Watchdg_tim_ctl,
        Watchdg_tim_val,
        Timestp_ctl,
        Sec_timestp,
        Min_timestp,
        Hour_timestp,
        Day_timestp,
        Mon_timestp,
        Year_timestp,
        Aging_offset,
        RAM_addr_MSB,
        RAM_addr_LSB,
        RAM_wrt_cmd,
        RAM_rd_cmd
    }
    RegisterName;

    /** Error code */
    typedef enum {
        NO_ERROR                = 0,
        CLOCK_INTEGRITY_FAIL    = 1,
        I2C_ACCESS_FAIL         = 2,
        TIME_FUNC_ERROR         = ((time_t)-1)
    }
    ErrorNum;

    /** Create a PCF2127 instance connected to specified I2C pins with specified address
     *
     *  @param I2C_sda      I2C-bus SDA pin
     *  @param I2C_scl      I2C-bus SCL pin
     *  @param vControl_1   (option) data for Control_1 register (default setting generates interrupts by second and minute)
     *  @param vControl_2   (option) data for Control_2 register
     *  @param vControl_3   (option) data for Control_3 register (default setting of battery switch-over function as standard mode)
     *  @param vCLKOUT_ctl  (option) data for CLKOUT_ctl register (default setting 1Hz output on CLKOUT pin)
     CLKOUT_ctl
     */
    PCF2127( PinName sda, PinName sdl, char vControl_1 = Cntl1, char vControl_2 = Cntl2, char vControl_3 = Cntl1, char vCLKOUT_ctl = ClkOut );

    /** Create a PCF2127 instance connected to specified I2C pins with specified address
     *
     *  @param i2c          I2C object (instance)
     *  @param vControl_1   (option) data for Control_1 register (default setting generates interrupts by second and minute)
     *  @param vControl_2   (option) data for Control_2 register
     *  @param vControl_3   (option) data for Control_3 register (default setting of battery switch-over function as standard mode)
     *  @param vCLKOUT_ctl  (option) data for CLKOUT_ctl register (default setting 1Hz output on CLKOUT pin)
     */
    PCF2127( I2C &i2c, char vControl_1 = Cntl1, char vControl_2 = Cntl2, char vControl_3 = Cntl1, char vCLKOUT_ctl = ClkOut );

    /** Destractor
     */
    ~PCF2127();

    /** Clock integrity check
     *
     *  @return     non-zero value if the clock was stopped (means need to set the time)
     */
    int     is_init_required( void );

    /** Set the time
     *
     *  @param dtp  Pointer to struct tm
     *  @return     Error code (NO_ERROR==0)
     */
    int     set_time( struct tm *dtp );

    /** Set the time
     *
     *  @param tp   pointer to time_t
     *  @return     Error code (NO_ERROR==0)
     */
    int     set_time( time_t *tp );

    /** Set the time
     *
     *  @param s    String data: The time information should be given in format of "YYYY MM DD HH MM SS"
     *  @return     Error code (NO_ERROR==0)
     */
    int     set_time( char *s );

    /** Get time of day
     *
     *  This function works similar to "time()" in standard-C-library
     *
     *  @param tp   Pointer to time_t
     *  @return     Error code (NO_ERROR==0)
     */
    time_t  time( time_t *tp );

    /** Register access interface with integer to BCD conversion
     *
     *  @param addr Register address
     *  @param s    Integer data. Converted to BCD before writing inot the register.
     *  @return     Error code (NO_ERROR==0)
     */
    int     set_alarm( char addr, char s );

    /** Clear interrupt flag
     *
     *  Clears interrupt flags by writing 0x00 into Control_2 register
     *
     *  @return Error code (NO_ERROR==0)
     */
    int     clear_intr( void );

    /** Writing data into RAM (for PCF2127 only)
     *
     *  Write data into PCF2127 internal RAM
     *
     *  @param address  target address of internal RAM
     *  @param *p       pointer to write data buffer
     *  @param size     size of writing data
     *  @return         Error code (NO_ERROR==0)
     */
    int     RAM_write( int address, char *p, int size );

    /** Reading data from RAM (for PCF2127 only)
     *
     *  Read data from PCF2127 internal RAM
     *
     *  @param address  target address of internal RAM
     *  @param *p       pointer to read data buffer
     *  @param size     size of writing data
     *  @return         Error code (NO_ERROR==0)
     */
    int     RAM_read( int address, char *p, int size );

private:

    typedef enum {
        Cntl1   = 0x03,
        Cntl2   = 0x00,
        Cntl3   = 0x00,
        ClkOut  = 0x46
    }
    DefaultRegParam;

    int     init( char vControl_1, char vControl_2, char vControl_3, char vCLKOUT_ctl );
    int     set_register( char addr, char data );
    int     read_register( char addr );
    char    i2bcd( char n );
    char    bcd2i( char bcd );
    int     set_RAM_address( char address );
    
    I2C     *i2c_p;
    I2C     &i2c;
    char    device_address;
}
;
#endif  // end of "#ifndef MBED_PCF2127"
