/** PCA(L)9554 8-bit I2C-bus GPIO expander
 *
 *  An operation sample of PCA(L)9554 and PCA9538.
 *  mbed accesses the PCAL9554 registers through I2C.
 *
 *  @class   PCAL9554
 *  @author  Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version 0.6
 *  @date    19-Mar-2015
 *
 *  Released under the Apache 2 license
 *
 *  About PCAL9554:
 *    http://www.jp.nxp.com/products/interface_and_connectivity/i2c/i2c_general_purpose_i_o/series/PCAL9554B_PCAL9554C.html
 */

#ifndef     MBED_PCAL9554
#define     MBED_PCAL9554

#include    "mbed.h"
#include    "PCAL955x.h"
#include    "CompGpioExpAPI.h"

/** PCAL9554 class
 *
 *  This is a driver code for the Low-voltage 8-bit I2C-bus GPIO with Agile I/O.
 *  This class provides interface for PCAL9554 operation.
 *  Detail information is available on next URL.
 *    http://www.jp.nxp.com/products/interface_and_connectivity/i2c/i2c_general_purpose_i_o/series/PCAL9554B_PCAL9554C.html
 *
 *  
 *  PCAL9554 library's basic IO operation is compatible to PCA9554, PCA9538.
 *  This library can be used for those GPIO expander chips also.
 *  Next sample code shows operation based on low-level-API (operated by just device instane)
 *  
 *  Example:
 *  @code
 *  //  GPIO-expander operation sample using a device instance
 *  
 *  #include    "mbed.h"
 *  #include    "PCAL9555.h"
 *
 *  PCAL9555    gpio( p28, p27, 0xE8 );     //  using PCA9539
 *
 *  int main() {
 *      gpio.configure( 0xFFFF );           //  Set all pins: input
 *      printf( "  0x%04X\r\n", (int)gpio );//  Print pins state
 *
 *      gpio.configure( 0x0000 );           //  Set all pins: output
 *      int count   = 0;
 *      while(1) {
 *          gpio.write( count++ );
 *      }
 *  }
 *  @endcode
 *
 *  GpioDigitalInOut, GpioDigitalOut, GpioDigitalIn,
 *  GpioBusInOut, GpioBusOut and GpioBusIn API class are available also.
 *  For those API details, please find those class library page.
 *  The GpioDigital* and GpioBus* APIs can be used as next sample code.
 *
 *  @code
 *  //  GPIO-expander operation sample using high-level-API
 *
 *  #include    "mbed.h"
 *  #include    "PCAL9554.h"
 *
 *  PCAL9555        gpio( p28, p27, 0xE0 );     //  using PCA9538
 *
 *  //  The GPIO pins are grouped in some groups and operated as bus I/O
 *  GpioBusIn       bus_in( gpio, X0, X1, X2, X3 );
 *  GpioBusOut      bus_out( gpio, X4, X5, X6 );
 *  GpioDigitalOut  myled( gpio, X7 );
 *
 *  int main() {
 *      int count   = 0;
 *      while(1) {
 *          printf( "In  = 0x%01X\r\n", (int)bus_in );
 *          bus_out = count;
 *          myled   = count & 0x1;
 *          count++;
 *          wait( 0.1 );
 *      }
 *  }
 *  @endcode
 */
class PCAL9554 : public PCAL955x
{

public:
    /** Name of the PCAL9554 registers */
    enum command_reg {
        InputPort               = 0x00, /**< InputPort register                 */
        OutoutPort,                     /**< OutoutPort register                */
        PolarityInversionPort,          /**< PolarityInversionPort register     */
        ConfigurationPort,              /**< ConfigurationPort register         */
        OutputDriveStrength0    = 0x40, /**< OutputDriveStrength0 register      */
        OutputDriveStrength1,           /**< OutputDriveStrength1 register      */
        InputLatch,                     /**< InputLatch register                */
        PullUpPullDowmEnable,           /**< PullUpPullDowmEnable register      */
        PullUpPullDowmSelection,        /**< PullUpPullDowmSelection register   */
        InterruptMask,                  /**< InterruptMask register             */
        InterruptStatus,                /**< InterruptStatus register           */
        OutputPortConfiguration = 0x4F, /**< OutputPortConfiguration register   */
    };

#if DOXYGEN_ONLY
    /** GPIO-Expander pin names
     *    for when the high-level APIs 
     *    (GpioDigitalOut, GpioDigitalInOut, GpioDigitalIn, 
     *    GpioBusOut, GpioBusInOut are GpioBusIn) are used
     */
    typedef enum {
        X0_0,           /**< P0_0 pin */
        X0_1,           /**< P0_1 pin */
        X0_2,           /**< P0_2 pin */
        X0_3,           /**< P0_3 pin */
        X0_4,           /**< P0_4 pin */
        X0_5,           /**< P0_5 pin */
        X0_6,           /**< P0_6 pin */
        X0_7,           /**< P0_7 pin */
        X0  = X0_0,     /**< P0_0 pin */
        X1  = X0_1,     /**< P0_1 pin */
        X2  = X0_2,     /**< P0_2 pin */
        X3  = X0_3,     /**< P0_3 pin */
        X4  = X0_4,     /**< P0_4 pin */
        X5  = X0_5,     /**< P0_5 pin */
        X6  = X0_6,     /**< P0_6 pin */
        X7  = X0_7,     /**< P0_7 pin */

        X_NC = ~0x0L    /**< for when the pin is left no-connection */
    } GpioPinName;
#endif

    /** Create a PCAL9554 instance connected to specified I2C pins with specified address
     *
     * @param i2c_sda       I2C-bus SDA pin
     * @param i2c_sda       I2C-bus SCL pin
     * @param i2c_address   I2C-bus address (default: 0x40)
     */
    PCAL9554( PinName i2c_sda, PinName i2c_scl, char i2c_address = PCAL955x::DEFAULT_I2C_ADDR );

    /** Create a PCAL9554 instance connected to specified I2C pins with specified address
     *
     * @param i2c_obj       I2C object (instance)
     * @param i2c_address   I2C-bus address (default: 0x40)
     */
    PCAL9554( I2C &i2c_obj, char i2c_address = PCAL955x::DEFAULT_I2C_ADDR );

    /** Destractor
     */
    virtual         ~PCAL9554();

    /** Returns the number of I/O pins
     *
     *  @returns
     *    The number of I/O pins
     */
    virtual int     number_of_pins( void );

#if DOXYGEN_ONLY

    /** Set output port bits
     *
     *  @param bit_pattern  16-bit output pattern for port1 and port0.
     *
     *  @note
     *    The data for pins, given as integer.
     *    The 16-bit MSB goes to P1_7 pin and LSB goes to P0_0 pin.
     *    Data will not come out from the pin if it is configured as input.
     *
     *  @see configure()
     */
    void            write( int bit_pattern );

    /** Read pin states
     *
     *  @return
     *    16-bit pattern from port1 and port0.
     *
     *  @note
     *    The data from pins, given as integer.
     *    The 16-bit port data comes from IO pins, P1_7 as MSB, P0_0 as LSB.
     *    Data cannot be read if the port is configured as output.
     *
     *  @see configure()
     */
    int             read( void );

    /** Polarity setting
     *
     *  @param bit_pattern  16-bit polarity setting pattern for port1 and port0.
     *    If the bit is set to '1', the state will be inverted.
     *    (Default state is all '0')
     *
     *  @see configure()
     */
    void            polarity( int bit_pattern );

    /** Set IO congiguration
     *
     *  @param bit_pattern  16-bit IO direction setting pattern for port1 and port0.
     *
     *  @note
     *    The data for pins, given as integer.
     *    The 16-bit MSB goes to P1_7 pin and LSB goes to P0_0 pin.
     *    Data will not come out from the pin if it is configured as input.
     *
     *  @see write()
     *  @see read()
     */
    void            configure( int bit_pattern );

    /** Set interrupt mask
     *
     *  @param bit_pattern  16-bit interrupt mask
     *
     *  @see interrupt_status()
     */
    void            interrupt_mask( int bit_pattern );

    /** Read interrupt status
     *
     *  @return
     *    16-bit data from interrupt status registers
     *
     *  @see interrupt_status()
     */
    int             interrupt_status( void );

    /** A shorthand for read()
     */
    operator int( void );

#endif

    /** Write 16-bit data into registers
     *
     *  @param reg_index    Direst access to registers.
     *    The registers can be referenced by regiser index
     *  @param data  16-bit data. Data will be written into two 8-bit registers
     */
    virtual void    reg_index_write( char register_index, int data );

    /** Read 16-bit data from registers
     *
     *  @param reg_index    Direst access to registers.
     *    The registers can be referenced by regiser index
     *
     *  @return
     *    16-bit data from two registers.
     *    The register which has lower address will be upper 8-bit data.
     */
    virtual int     reg_index_read( char register_index );

    /** A shorthand for write()
     */
    PCAL9554&       operator= ( int bit_pattern );
    PCAL9554&       operator= ( PCAL9554& rhs );

private:

    static const char   regmap[];
    const int           n_of_pins;
}
;

#endif  //  MBED_PCAL9554
