/** GpioDigitalInOut API for GPIO-expander component class 
 *
 *  @author  Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version 0.6
 *  @date    19-Mar-2015
 *
 *  Released under the Apache 2 license
 */

#ifndef     MBED_GpioDigitalInOut
#define     MBED_GpioDigitalInOut

#include    "mbed.h"
#include    "CompGpioExp.h"
#include    "GpioDigitalInOut.h"

/** GpioDigitalInOut class
 *
 *  @class GpioDigitalInOut
 *
 *  "GpioDigitalInOut" class works like "DigitalInOut" class of mbed-SDK. 
 *  This class provides pin oriented API, abstracting the GPIO-expander chip. 
 *
 *  Example:
 *  @code
 *  #include "mbed.h"
 *  #include "PCAL9555.h"
 *  
 *  PCAL9555            gpio_exp( p28, p27, 0x40 );    //  SDA, SCL, Slave_address(option)
 *  GpioDigitalInOut    pin( gpio_exp, X0_0 );
 *  
 *  int main()
 *  {
 *      pin.output();
 *      pin = 0;
 *      wait_us( 500 );
 *      pin.input();
 *      wait_us( 500 );
 *  }
 *  @endcode
 */
class GpioDigitalInOut
{
public:

#if DOXYGEN_ONLY
    /** GPIO-Expander pin names */
    typedef enum {
    X0_0,           /**< P0_0 pin                                   */
    X0_1,           /**< P0_1 pin                                   */
    X0_2,           /**< P0_2 pin                                   */
    X0_3,           /**< P0_3 pin                                   */
    X0_4,           /**< P0_4 pin                                   */
    X0_5,           /**< P0_5 pin                                   */
    X0_6,           /**< P0_6 pin                                   */
    X0_7,           /**< P0_7 pin                                   */
    X1_0,           /**< P1_0 pin (for 16-bit GPIO device only)     */
    X1_1,           /**< P1_1 pin (for 16-bit GPIO device only)     */
    X1_2,           /**< P1_2 pin (for 16-bit GPIO device only)     */
    X1_3,           /**< P1_3 pin (for 16-bit GPIO device only)     */
    X1_4,           /**< P1_4 pin (for 16-bit GPIO device only)     */
    X1_5,           /**< P1_5 pin (for 16-bit GPIO device only)     */
    X1_6,           /**< P1_6 pin (for 16-bit GPIO device only)     */
    X1_7,           /**< P1_7 pin (for 16-bit GPIO device only)     */
    X0  = X0_0,     /**< P0_0 pin                                   */
    X1  = X0_1,     /**< P0_1 pin                                   */
    X2  = X0_2,     /**< P0_2 pin                                   */
    X3  = X0_3,     /**< P0_3 pin                                   */
    X4  = X0_4,     /**< P0_4 pin                                   */
    X5  = X0_5,     /**< P0_5 pin                                   */
    X6  = X0_6,     /**< P0_6 pin                                   */
    X7  = X0_7,     /**< P0_7 pin                                   */
    X8  = X1_0,     /**< P1_0 pin (for 16-bit GPIO device only)     */
    X9  = X1_1,     /**< P1_1 pin (for 16-bit GPIO device only)     */
    X10 = X1_2,     /**< P1_2 pin (for 16-bit GPIO device only)     */
    X11 = X1_3,     /**< P1_3 pin (for 16-bit GPIO device only)     */
    X12 = X1_4,     /**< P1_4 pin (for 16-bit GPIO device only)     */
    X13 = X1_5,     /**< P1_5 pin (for 16-bit GPIO device only)     */
    X14 = X1_6,     /**< P1_6 pin (for 16-bit GPIO device only)     */
    X15 = X1_7,     /**< P1_7 pin (for 16-bit GPIO device only)     */
    
    X_NC = ~0x0L    /**< for when the pin is left no-connection     */
    } GpioPinName;
#endif

    /** Create a GpioDigitalInOut connected to the specified pin
     *
     *  @param gpiop    Instance of GPIO expander device
     *  @param pin_name DigitalInOut pin to connect to
     */
    GpioDigitalInOut( CompGpioExp &gpiop, GpioPinName pin_name );

    /**
     *  Destractor
     */
    virtual ~GpioDigitalInOut();

    /** Set the output, specified as 0 or 1 (int)
     *
     *  @param v    An integer specifying the pin output value,
     *      0 for logical 0, 1 (or any other non-zero value) for logical 1
     */
    virtual void    write( int v );

    /** Return the output setting, represented as 0 or 1 (int)
     *
     *  @returns
     *    an integer representing the output setting of the pin if it is an output,
     *    or read the input if set as an input
     */    
    virtual int     read( void );

    /** Set as an output
     */
    void            output( void );

    /** Set as an input
     */
    void            input( void );

    /** A shorthand for write()
     */
    GpioDigitalInOut&      operator= ( int rhs );
    GpioDigitalInOut&      operator= ( GpioDigitalInOut& rhs );

    /** A shorthand for read()
     */
    virtual operator int( void );

private:
    CompGpioExp     *gpio_p;
    GpioPinName     pin;
    int             output_state;
    int             config_state;

    void    write( int pin, int value );
    void    configure( int pin, int value );
}
;

#endif  //  MBED_GpioDigitalInOut
