/** PCA9632 PWM control LED driver
 *
 *  An operation sample of PCA9632 16-channel Fm+ I2C-bus 100mA/40V LED driver.
 *  mbed accesses the PCA9632 registers through I2C.
 *
 *  @author  Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version 0.5
 *  @date    4-Mar-2015
 *
 *  Released under the Apache 2 license
 *
 *  About PCA9632:
 *    http://www.nxp.com/products/lighting_driver_and_controller_ics/i2c_led_display_control/series/PCA9632.html
 */

#ifndef     MBED_PCA9632
#define     MBED_PCA9632

#include    "mbed.h"
#include    "CompLedDvr.h"
#include    "LedPwmOut.h"

#define     ALLPORTS        0xFF


/** PCA9632 class
 *
 *  @class   PCA9632
 *
 *  This is a driver code for the PCA9632 is an I2C-bus controlled 4-bit LED driver
 *  optimized for Red/Green/Blue/Amber (RGBA) color mixing applications. In Individual
 *  brightness control mode, each LED output has its own 8-bit resolution (256 steps)
 *  fixed frequency Individual PWM controller that operates at 1.5625 kHz with a duty
 *  cycle that is adjustable from 0 % to 99.6 % to allow the LED to be set to a specific
 *  brightness value.
 *  This class provides interface for PCA9632 operation and accessing its registers.
 *  Detail information is available on next URL.
 *    http://www.jp.nxp.com/products/interface_and_connectivity/i2c/i2c_led_display_control/series/PCA9632.html
 *
 *  Example:
 *  @code
 *  #include "mbed.h"
 *  #include "PCA9632.h"
 *
 *  PCA9632     led_cntlr( p28, p27, 0xC4 );    //  SDA, SCL, Slave_address(option)
 *  LedPwmOut   led( led_cntlr, L0 );           //  for LED0 pin
 *
 *  int main()
 *  {
 *      //
 *      //  Here are two types of PWM control samples
 *      //  (User can choose one of those interface to set the PWM.)
 *      //
 *      //  1st sample is using LedPwmOut API.
 *      //    It provides similar interface like PwmOut of mbed-SDK
 *      //
 *      //  2nd sample is using PCA9632 class function.
 *      //    the 'pwm()' function takes LED channel number and duty-ratio value
 *      //
 *
 *      while ( 1 ) {
 *
 *          //
 *          //  1st sample is using LedPwmOut API.
 *          //  PWM control via LedPwmOut
 *          //
 *          for ( int i = 0; i < 3; i++ ) {
 *              for( float p = 0.0f; p < 1.0f; p += 0.1f ) {
 *                  led     = p;  //  Controls LED0 pin
 *                  wait( 0.1 );
 *              }
 *          }
 *
 *          //
 *          //  2nd sample is using PCA9632 class function.
 *          //  PWM control by device class function call
 *          //
 *          for ( int i = 0; i < 3; i++ ) {
 *              for( float p = 0.0f; p < 1.0f; p += 0.1f ) {
 *                  led_cntlr.pwm( 1, p );  //  Controls LED1 pin
 *                  wait( 0.1 );
 *              }
 *          }
 *      }
 *  }
 *  @endcode
 */
class PCA9632 : public CompLedDvr
{
public:

#if DOXYGEN_ONLY
    /** PCA9626 pin names high-level API i.e. LedPwmOut */
    typedef enum {
        L0,            /**< LED0 pin                               */
        L1,            /**< LED2 pin                               */
        L2,            /**< LED2 pin                               */
        L3,            /**< LED2 pin                               */
        L_NC = ~0x0L   /**< for when the pin is left no-connection */
    } LedPinName;
#endif // DOXYGEN_ONLY

    /** Name of the PCA9632 registers (for direct register access) */
    enum command_reg {
        MODE1,      /**< MODE1 register      */
        MODE2,      /**< MODE2 register      */
        PWM0,       /**< PWM0 register       */
        PWM1,       /**< PWM1 register       */
        PWM2,       /**< PWM2 register       */
        PWM3,       /**< PWM3 register       */
        GRPPWM,     /**< GRPPWM register     */
        GRPFREQ,    /**< GRPFREQ register    */
        LEDOUT,     /**< LEDOUT register    */
        SUBADR1,    /**< SUBADR1 register    */
        SUBADR2,    /**< SUBADR2 register    */
        SUBADR3,    /**< SUBADR3 register    */
        ALLCALLADR, /**< ALLCALLADR register */

        REGISTER_START          = MODE1,
        PWM_REGISTER_START      = PWM0,
    };

    /** Difinition of the number of LED pins */
    enum {
        N_OF_PORTS   = 4
    };

    /** Create a PCA9632 instance connected to specified I2C pins with specified address
     *
     * @param i2c_sda       I2C-bus SDA pin
     * @param i2c_sda       I2C-bus SCL pin
     * @param i2c_address   I2C-bus address (default: 0xC4)
     */
    PCA9632( PinName i2c_sda, PinName i2c_scl, char i2c_address = DEFAULT_I2C_ADDR );

    /** Create a PCA9632 instance connected to specified I2C pins with specified address
     *
     * @param i2c_obj       I2C object (instance)
     * @param i2c_address   I2C-bus address (default: 0xC4)
     */
    PCA9632( I2C &i2c_obj, char i2c_address = DEFAULT_I2C_ADDR );

    /** Destractor
     */
    virtual ~PCA9632();

    /** Performs Software reset via I2C bus
     */
    void    reset( void );

    /** Set the output duty-cycle, specified as a percentage (float)
     *
     * @param port  Selecting output port
     *    'ALLPORTS' can be used to set all port duty-cycle same value.
     * @param v     A floating-point value representing the output duty-cycle,
     *    specified as a percentage. The value should lie between
     *    0.0f (representing on 0%) and 1.0f (representing on 99.6%).
     *    Values outside this range will have undefined behavior.
     */
    virtual void    pwm( int port, float v );

    /** Set all output port duty-cycle, specified as a percentage (array of float)
     *
     * @param vp    Aray to floating-point values representing the output duty-cycle,
     *    specified as a percentage. The value should lie between
     *    0.0f (representing on 0%) and 1.0f (representing on 99.6%).
     *
     *  @note
     *    The aray should have length of 4
     */
    void    pwm( float *vp );

    /** Register write (single byte) : Low level access to device register
     *
     * @param reg_addr  Register address
     * @param data      Value for setting into the register
     */
    void    write( char reg_addr, char data );

    /** Register write (multiple bytes) : Low level access to device register
     *
     * @param data      Pointer to an array. First 1 byte should be the writing start register address
     * @param length    Length of data
     */
    void    write( char *data, int length );

    /** Register read (single byte) : Low level access to device register
     *
     * @param reg_addr  Register address
     * @return          Read value from register
     */
    char    read( char reg_addr );

    /** Register write (multiple bytes) : Low level access to device register
     *
     * @param reg_addr  Register address
     * @param data      Pointer to an array. The values are stored in this array.
     * @param length    Length of data
     */

    void    read( char reg_addr, char *data, int length );

protected:
    enum {
        DEFAULT_I2C_ADDR    = 0xC4,
        AUTO_INCREMENT      = 0x80,
        PWMALL              = 0xFF
    };

private:
    void    initialize( void );

    I2C     *i2c_p;
    I2C     &i2c;
    char    address;    //  I2C slave address
}
;

#endif  //  MBED_PCA9632
