/** A sample code for PCA9629
 *
 *  @author  Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version 1.1
 *  @date    23-Jul-2012
 *
 *  revision history
 *      version 1.0 (24-Apr-2011) : Initial version
 *      version 1.1 (23-Jul-2012) : API modification
 *                                  Correction for comments
 *
 *  Released under the MIT License: http://mbed.org/license/mit
 *
 *  An operation sample of PCU9629 stepper motor controller.
 *  The mbed accesses the PCU9629 registers through I2C.
 *
 *  This sample code demonstrates 4 type of PCA9629 operation
 *
 *  About PCA9629:
 *    http://www.nxp.com/products/interface_and_connectivity/i2c/i2c_bus_controller_and_bridge_ics/PCA9629PW.html
 */

#include "mbed.h"
#include "PCA9629.h"

BusOut      leds( LED4, LED3, LED2, LED1 );
PCA9629     motor_controller( p28, p27, 48 );  //  SDA, SCL, Steps/rotation, I2C_address (option)

void motor_action_0( void );
void motor_action_1( void );
void motor_action_2( void );
void motor_action_3( void );
void motor_action_CDE( void );

int main() {
    printf( "PCA9629 simple sample program\r\n" );

    while ( 1 ) {
        leds    = 0x1;
        motor_action_0();   //  demo 0

        leds    = 0x3;
        motor_action_1();   //  demo 1

        leds    = 0x7;
        motor_action_2();   //  demo 2

        leds    = 0xF;
        motor_action_3();   //  demo 3
    }
}

void motor_action_0( void ) {

    //  simple motor operation
    //  the motor spins 2 times (200pps) for clockwise (CW), 1 time (100pps) for counter-clockwise.

    motor_controller.stop();
    motor_controller.init_registers();

    motor_controller.pps( PCA9629::CW, 200 );
    motor_controller.rotations_and_steps( PCA9629::CW, 0, 96 );

    motor_controller.pps( PCA9629::CCW, 100 );
    motor_controller.rotations_and_steps( PCA9629::CCW, 0, 48 );

    for ( int i = 0; i < 2; i++ ) {
        motor_controller.start( PCA9629::CW );
        wait( 2.0 );

        motor_controller.start( PCA9629::CCW );
        wait( 2.0 );
    }
}

void motor_action_1( void ) {

    //  ramp control demo

    motor_controller.stop();
    motor_controller.init_registers();

    //  auto generated example code for PCA9629 on mbed
    //  setting: total operation time = 2, initial speed = 40.69, rotations = 5, steps = 0, ramp-up enable = enable, ramp-down enable = enable, rotate direction = CW

    //  motor_controller.write( PCA9629::STEPS_PER_ROATION, 48  );
    motor_controller.write( PCA9629::CW__STEP_WIDTH, 1175  );
    motor_controller.write( PCA9629::CW__STEP_COUNT, 35  );
    motor_controller.write( PCA9629::CW__ROTATION_COUNT, 2  );
    motor_controller.write( PCA9629::RAMPCNTL, 0x37  );

    for ( int i = 0; i < 2; i++ ) {
        motor_controller.write( PCA9629::MCNTL, 0xA8  );  //  start
        wait( 2.5 );
    }
}

void motor_action_2( void ) {

    //  interrupt operation demo

    motor_controller.stop();

    //  this array is a sample for PCA9629 local interrupt operation
    //  the P0 and P1 inputs enabled for interrupt in.
    //  motor rotation will be reversed by interrupt signals.
    //  the motor direction will be changed 12 steps after when the interrupt happened
    //
    //  CW = 40.69pps, CCW = 81.38pps

    char init_array2[] = {    0x80,
                              0x20, 0xE2, 0xE4, 0xE6, 0xE0, 0xFF, 0x10,             //  for registers MODE - WDCNTL (0x00 - 0x06
                              0x00, 0x00,                                           //  for registers IP and INTSTAT (0x07, 0x08)
                              0x0F, 0x03, 0x0C, 0x0F, 0x03, 0x01, 0x03, 0x03, 0x01, //  for registers OP - INT_AUTO_CLR (0x09 - 0x11)
                              0x00, 0x00, 0x30, 0x00, 0xFF, 0x1F, 0xFF, 0x0F,       //  for registers SETMODE - CCWPWH (0x12 - 0x19)
                              0xFF, 0xFF, 0xFF, 0xFF, 0x0A, 0x00, 0x0A, 0x00,       //  for registers CWSCOUNTL - CCWRCOUNTH (0x1A - 0x21)
                              0x0C, 0x0C,                                           //  for registers EXTRASTEPS0 and EXTRASTEPS1 (0x22, 0x23)
                              0x00, 0x00, 0xA8                                      //  for registers RMPCNTL - MCNTL (0x24 - 0x26)
                         };

    motor_controller.set_all_registers( init_array2, sizeof( init_array2 ));
    wait( 5.0 );

    motor_controller.stop();
    wait( 0.5 );

}

void motor_action_3( void ) {

    //  speed change sample while the motor is running

    motor_controller.stop();
    motor_controller.init_registers();

    motor_controller.rotations( PCA9629::CCW, 0xFFFF );

    for ( int i = 0; i < 2; i++ ) {
        for ( int s = 1; s <= 5; s++ ) {
            motor_controller.pps( PCA9629::CCW, 55 * s );
            motor_controller.start( PCA9629::CCW );
            wait( 1.0 );
            motor_controller.stop();
        }
    }
}

void motor_action_CDE( void ) {
    char    scale[ 8 ]  = { 4, 6, 8, 9, 11, 13, 15, 16 };  //   CDEFGABC (A=0)
    float   freq[ 8 ];

    for ( int i = 0; i < 8; i++ )
        freq[ i ]   = (440.0 / 8.0) * pow( 2.0, (float)(scale[ i ]) / 12.0 );

    motor_controller.stop();
    motor_controller.init_registers();
    motor_controller.rotations( PCA9629::CCW, 0xFFFF );
    motor_controller.write( PCA9629::PHCNTL, 0x1 );

    for ( int i = 0; i < 2; i++ ) {
        motor_controller.stop();
        for ( int s = 0; s < 8; s++ ) {
            motor_controller.pps( PCA9629::CCW, freq[ s ] );
            motor_controller.start( PCA9629::CCW );
            wait( 1 );

            motor_controller.stop();
        }
    }
}

