/** A sample code for PCA9629A
 *
 *  @author  Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version 1.1
 *  @date    12-Sep-2012
 *
 *  revision history (PCA9629A)
 *      version 0.1 (05-Jun-2013) : PCA9629"A" initial version
 *      version 0.2 (05-Jun-2013) : register name fix, register description added
 *      version 1.0 (23-Apr-2014) : sample code for revision 1.0 library
 *      version 1.1 (12-Sep-2014) : sample code for revision 1.1 library
 *
 *  Released under the Apache 2 license License
 *
 *  An operation sample of PCU9629A stepper motor controller.
 *  The mbed accesses the PCU9629A registers through I2C.
 *
 *  This sample code demonstrates 4 type of PCA9629A operation
 *
 *  About PCA9629A:
 *    http://www.nxp.com/products/interface_and_connectivity/i2c/i2c_bus_controller_and_bridge_ics/PCA9629APW.html
 */

#include "mbed.h"
#include "PCA9629A.h"

BusOut      leds( LED4, LED3, LED2, LED1 );

PCA9629A    motor_controller( p28, p27, 0x42 );  //  SDA, SCL, Steps/rotation, I2C_address (option)

void motor_action_0( void );
void motor_action_1( void );
void motor_action_2( void );
void motor_action_3( void );
void motor_action_CDE( void );

int main()
{
    printf( "PCA9629A simple sample program\r\n" );

    while ( 1 ) {
        leds    = 0x1;
        motor_action_0();   //  demo 0

        leds    = 0x3;
        motor_action_1();   //  demo 1

        leds    = 0x7;
        motor_action_2();   //  demo 2

        leds    = 0xF;
        motor_action_3();   //  demo 3
    }
}

void motor_action_0( void )
{
    //  simple motor operation
    //  the motor spins 2 times (200pps) for clockwise (CW), 1 time (100pps) for counter-clockwise.

    motor_controller.stop();
    motor_controller.init_registers();

    motor_controller.pps( PCA9629A::CW, 200 );
    motor_controller.steps( PCA9629A::CW, 96 );

    motor_controller.pps( PCA9629A::CCW, 100 );
    motor_controller.steps( PCA9629A::CCW, 48 );

    for ( int i = 0; i < 2; i++ ) {
        motor_controller.start( PCA9629A::CW );
        wait( 2.0 );

        motor_controller.start( PCA9629A::CCW );
        wait( 2.0 );
    }
}

void motor_action_1( void )
{
    //  ramp control demo

    motor_controller.stop();
    motor_controller.init_registers();

    motor_controller.write( PCA9629A::RUCNTL, 0x27  );
    motor_controller.write( PCA9629A::RDCNTL, 0x27  );
    motor_controller.write( PCA9629A::CW__STEP_WIDTH, 1175  );
    motor_controller.write( PCA9629A::CW__STEP_COUNT, 131  );

    for ( int i = 0; i < 2; i++ ) {
        motor_controller.start( PCA9629A::CW );  //  start
        wait( 2.5 );
    }
}

void motor_action_2( void )
{
    //  interrupt operation demo

    motor_controller.stop();

    //  this array is a sample for PCA9629A local interrupt operation
    //  the P0 and P1 inputs enabled for interrupt in.
    //  motor rotation will be reversed by interrupt signals.
    //  the motor direction will be changed 12 steps after when the interrupt happened
    //
    //  CW = 40.69pps, CCW = 81.38pps

    char init_array2[] = {    0x80,
                              0x20, 0x0A, 0x00, 0x03, 0x13, 0x1C,             //  for registers MODE - MSK (0x00 - 0x07
                              0x00, 0x00, 0x69, 0x00, 0x00,                   //  for registers INTSTAT - EXTRASTEPS1 (0x06, 0xA)
                              0x10, 0x80,                                     //  for registers OP_CFG_PHS and OP_STAT_TO (0x0B - 0xC)
                              0x09, 0x09, 0x00, 0x00, 0x00,                   //  for registers RUCNTL - LOOPDLY_CCW (0xD- 0x10)
                              0x60, 0x00, 0x60, 0x00, 0x05, 0x0D, 0x05, 0x0D, //  for registers CWSCOUNTL - MCNTL (0x12 - 0x1A)
                              0x82,                                           //  for register MCNTL (0x1A)
                              0xE2, 0xE4, 0xE6, 0xE0                          //  for registers SUBADR1 - ALLCALLADR (0x1B - 0x1E)
                         };

    motor_controller.set_all_registers( init_array2, sizeof( init_array2 ));
    wait( 5.0 );

    motor_controller.stop();
    wait( 0.5 );
}

void motor_action_3( void )
{
    //  speed change sample while the motor is running

    motor_controller.stop();
    motor_controller.init_registers();

    motor_controller.steps( PCA9629A::CCW, 0xFFFF );

    for ( int i = 0; i < 2; i++ ) {
        for ( int s = 1; s <= 5; s++ ) {
            motor_controller.pps( PCA9629A::CCW, 55 * s );
            motor_controller.start( PCA9629A::CCW );
            wait( 1.0 );
        }
    }
}

void motor_action_CDE( void )
{
    char    scale[ 8 ]  = { 4, 6, 8, 9, 11, 13, 15, 16 };  //   CDEFGABC (A=0)
    float   freq[ 8 ];

    for ( int i = 0; i < 8; i++ )
        freq[ i ]   = (440.0 / 8.0) * pow( 2.0, (float)(scale[ i ]) / 12.0 );

    motor_controller.stop();
    motor_controller.init_registers();
    motor_controller.steps( PCA9629A::CCW, 0xFFFF );

    for ( int i = 0; i < 2; i++ ) {
        motor_controller.stop();
        for ( int s = 0; s < 8; s++ ) {
            motor_controller.pps( PCA9629A::CCW, freq[ s ] );
            motor_controller.start( PCA9629A::CCW );
            wait( 1 );
        }
    }
}
