/**
 *  A demo code for the PCA9622 LED 8x8 library
 *
 *  @author  Tedd OKANO
 *  @version 1.0
 *  @date    03-Dec-2014
 *
 *  This is a sample demo code for the PCA9622_LED8x8 library.
 *  Target hardware : "I2C 8x8 LED matrix board" from Switch Science
 *    https://www.switch-science.com/catalog/2071/
 *
 *  The I2C LED controller PCA9622 is used on this module
 *  that ebables to control the LEDs with PWM brightness control.
 *
 *  For more information about the PCA9622:
 *    http://www.nxp.com/documents/data_sheet/PCA9622.pdf
 */

#include "mbed.h"
#include "PCA9622_LED8x8.h"

//  Choose a target platform from next list
PCA9622_LED8x8  matrix( p28, p27 );     //  for 40pin type mbed
//PCA9622_LED8x8  matrix( D14, D15 );   //  for Arduino type mbed
//PCA9622_LED8x8  matrix( dp5, dp27 );  //  for mbed LPC1114

Ticker  _ticker;
int     demo_mode   = 0;

void    demo_mode_change();
float   func0( int x, int y, int t );    //  function to make 8x8 image
float   func1( int x, int y, int t );    //  function to make 8x8 image
float   func2( int x, int y, int t );    //  function to make 8x8 image
float   func3( int x, int y, int t );    //  function to make 8x8 image

typedef float (*func_ptr)( int x, int y, int t );

func_ptr    fp[]    = {
    func0,
    func1,
    func2,
    func3
};

int main()
{
    float   image[ 8 ][ 8 ];  //
    int     count   = 0;

    _ticker.attach( &demo_mode_change, 10 );
    matrix.start();

    while(1) {

        //  making 8x8 image to "image" array
        for ( int i = 0; i < 8; i++ )
            for ( int j = 0; j < 8; j++ )
                image[ i ][ j ]   = (*fp[ demo_mode % (sizeof( fp ) / sizeof( func_ptr )) ])( i, j, count );

        //  set the image into library internal bufer
        matrix.set_data( image );

        count++;
        wait( 0.05 );
    }
}

void    demo_mode_change()
{
    demo_mode++;
}

float func0( int x, int y, int t )
{
    //  NXP logo

    static char bm_nxp[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

        0xFF, 0xFF, 0xFF, 0xF0, 0x78, 0x3C, 0x1E, 0x0F,
        0xFF, 0x7E, 0xBD, 0xDB, 0xE7, 0x7E, 0x3C, 0x18,
        0x3C, 0x7E, 0xE7, 0xDB, 0xBD, 0x7E, 0xFF, 0xCC,
        0xCC, 0xCC, 0xCC, 0xCC, 0xFC, 0x78, 0x78,

        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    int     index;
    int     direction   = 0;

    direction   = (t / (sizeof( bm_nxp ) - 8)) & 0x1;
    index       = (t % (sizeof( bm_nxp ) - 8));
    index       = direction ? (sizeof( bm_nxp ) - 8) - index : index;
    index       = index + y;

    return (bm_nxp[ index ] >> (7 - x)) & 0x1;
}

float func1( int x, int y, int t )
{
    //  ripple

    const float display_offset          = 3.5;
//const float  display_offset         = 0.0;
    const float size                    = 0.3;
    const float eccentricity_cycle      = 0.0;
    const float eccentricity_amplitude  = 0.0;

    float   xx;
    float   yy;
    float   tt;

    float   s;

    xx  = ((float)x - display_offset + eccentricity_amplitude * sin( (float)t * eccentricity_cycle )) * size;
    yy  = ((float)y - display_offset + eccentricity_amplitude * cos( (float)t * eccentricity_cycle )) * size;
    tt  = sin( (float)t * 0.2 ) + sin( (float)t * 0.1 );

    s   = cos( powf( xx * xx + yy * yy, 0.5 ) - tt );
    return ( powf( s, 4.0 ) );
}

float func2( int x, int y, int t )
{
    //  moving dot

    const float display_offset          = 3.5;
//const float  display_offset         = 0.0;
    const float size                    = 0.3;
    const float eccentricity_cycle      = 0.25;
    const float eccentricity_amplitude  = 2.5;

    float   xx;
    float   yy;
    float   tt;
    float   s;
    float   d;


    xx  = ((float)x - display_offset + eccentricity_amplitude * sin( (float)t * eccentricity_cycle )) * size;
    yy  = ((float)y - display_offset + eccentricity_amplitude * cos( (float)t * eccentricity_cycle )) * size;
    tt  = (float)t * 0.2;
    d   = powf( xx * xx + yy * yy, 0.5 ) - tt * 0.0;
    d   = d == 0.0 ? 1e-12 : d;
    d   *= 4.0;

    s   = sin( d ) / d;
    return ( powf( s, 4.0 ) );
}

float func3( int x, int y, int t )
{
    //  ripple with offset

    const float display_offset          = 3.5;
//const float  display_offset         = 0.0;
    const float size                    = 0.3;
    const float eccentricity_cycle      = 0.25;
    const float eccentricity_amplitude  = 2.5;

    float   xx;
    float   yy;
    float   tt;

    float   s;

    xx  = ((float)x - display_offset + eccentricity_amplitude * sin( (float)t * eccentricity_cycle )) * size;
    yy  = ((float)y - display_offset + eccentricity_amplitude * cos( (float)t * eccentricity_cycle )) * size;
    tt  = (float)t * 0.2;

    s   = cos( powf( xx * xx + yy * yy, 0.5 ) - tt );
    return ( powf( s, 4.0 ) );
}

