/**
 *  PCA9622 LED 8x8 library
 *
 *  @author  Tedd OKANO
 *  @version 1.1.1
 *  @date    26-Feb-2015
 *
 *  Library for "I2C 8x8 LED matrix board" from Switch Science
 *    https://www.switch-science.com/catalog/2071/
 *
 *  The I2C LED controller PCA9622 is used on this module 
 *  that ebables to control the LEDs with PWM brightness control. 
 *  
 *  For more information about the PCA9622:
 *    http://www.nxp.com/documents/data_sheet/PCA9622.pdf
 */

#include    "mbed.h"
#include    "PCA9622_LED8x8.h"

PCA9622_LED8x8::PCA9622_LED8x8( PinName sda, PinName scl, char slave_adr, float fr )
    :
    i2c_p( new I2C( sda, scl ) ), 
    i2c( *i2c_p ), 
    address( slave_adr ),
    framerate( fr ),
    in_operation( false ),
    line_counter( 0 ),
    frame_counter( 0 ),
    buffer_switch_request( false ),
    outgoing_buffer( 0 )
{
    initialize();
}

PCA9622_LED8x8::PCA9622_LED8x8( I2C &i2c_obj, char slave_adr, float fr )
    :
    i2c_p( NULL ), 
    i2c( i2c_obj ),
    address( slave_adr ),
    framerate( fr ),
    in_operation( false ),
    line_counter( 0 ),
    frame_counter( 0 ),
    buffer_switch_request( false ),
    outgoing_buffer( 0 )
{
    initialize();
}

PCA9622_LED8x8::~PCA9622_LED8x8()
{
    if ( NULL != i2c_p )
        delete  i2c_p;
}

void PCA9622_LED8x8::initialize( void )
{
    char    init[ 2 ][ 3 ]   = {
        { 0x80, 0x00, 0x05 },   //  initialize MODE1 and MODE2 registers
        { 0x96, 0xAA, 0xAA }    //  initialize LEDOUT2 and LEDOUT3 registers
    };

    i2c.frequency( 400 * 1000 );
    i2c.write( address, init[ 0 ], sizeof( init[ 0 ] ) );
    i2c.write( address, init[ 1 ], sizeof( init[ 1 ] ) );
    
    start();
}

void PCA9622_LED8x8::frame_rate( float rate )
{
    int     previous_state;

    previous_state  = in_operation;

    stop();
    framerate  = rate;

    if ( previous_state )
        start();
}

void PCA9622_LED8x8::start( void )
{
    t.attach( this, &PCA9622_LED8x8::draw_a_line, 1.0 / (framerate * 8.0) );
    in_operation    = true;
}

void PCA9622_LED8x8::stop( void )
{
    t.detach();
    in_operation    = false;
}

void PCA9622_LED8x8::draw_a_line( void )
{
    char        write_data[ 13 ];

    if ( buffer_switch_request && !line_counter ) {
        //  when the scan start, and if the buffer switching is requested ping-pong bufer will be switched
        outgoing_buffer         = !outgoing_buffer;
        buffer_switch_request   = false;
    }

    write_data[  0 ]    = 0x80 | 0xA;   //  pointing PWM8 register with increment flag 
    //  write_data[  9 ]    = 0x00;     //  don't need to stuff any data because this register will be ignored (because global PWM setting is not used)
    //  write_data[ 10 ]    = 0x00;     //  don't need to stuff any data because this register will be ignored (because global PWM setting is not used)

    for ( int i = 0; i < 8; i++ )
        write_data[ i + 1 ]   = pattern[ outgoing_buffer ][ line_counter ][ i ];    //  PWM data are set to PWM8..PWM15 registers (driving ROW)

    //  A line below works fine on Cortex-M3 but not on Cortex-M0. So I needed to rewrite it
    //  *((unsigned short *)(write_data + 11))  = 0x0001 << (line_counter << 1);    //  channel 0 to 7 are used to ON/OFF the column

    //  A line above is rewritten as next 3 lines
    unsigned short  tmp     = 0x0001 << (line_counter << 1);
    write_data[ 11 ]    = tmp &  0xFF;
    write_data[ 12 ]    = tmp >> 8;

    i2c.write( address, write_data, sizeof( write_data ) );     //  I2C transfer

    line_counter    = (line_counter + 1) & 0x7;

    if ( line_counter )
        frame_counter++;
}

void PCA9622_LED8x8::set_data( float p[ 8 ][ 8 ] )
{
    for ( int i = 0; i < 8; i++ )
        for ( int j = 0; j < 8; j++ )
            pattern[ !outgoing_buffer /* store image to in-active side */ ][ i ][ 7 - j ]   = (char)(p[ i ][ j ] * 255.0);

    buffer_switch_request   = true; //  when the buffer filling done, raise this flag to switch the pin-pong buffer
}
