/**
 *  PCA9622 LED 8x8 library
 *
 *  @author  Tedd OKANO
 *  @version 1.1.1
 *  @date    26-Feb-2015
 *
 *  Library for "I2C 8x8 LED matrix board" from Switch Science
 *    https://www.switch-science.com/catalog/2071/
 *
 *  The I2C LED controller PCA9622 is used on this module
 *  This LED controller chip enables brightness control by PWM.
 *
 *  For more information about the PCA9622:
 *    http://www.nxp.com/documents/data_sheet/PCA9622.pdf
 */

#ifndef MBED_PCA9622_LED8x8_H
#define MBED_PCA9622_LED8x8_H

#include "mbed.h"

/** PCA9622_LED8x8 class
 *
 *  PCA9622_LED8x8: A library to control LED matrix using the PCA9622
 *
 *  PCA9622 is an I2C LED controller.
 *  This library operates LED 8x8 matrix through the PCA9622.
 *
 *  Example:
 *  @code
 *  #include "mbed.h"
 *  #include "PCA9622_LED8x8.h"
 *
 *  PCA9622_LED8x8  matrix( p28, p27 );     //  for 40pin type mbed
 *  //PCA9622_LED8x8  matrix( D14, D15 );   //  for Arduino type mbed
 *  //PCA9622_LED8x8  matrix( dp5, dp27 );  //  for mbed LPC1114
 *
 *  float func( float x, float y, float t );    //  function to make 8x8 image
 *
 *  int main()
 *  {
 *      float   image[ 8 ][ 8 ];  //
 *      int     count   = 0;
 *
 *      matrix.start();
 *
 *      while(1) {
 *
 *          //  making 8x8 image to "image" array
 *          for ( int i = 0; i < 8; i++ )
 *              for ( int j = 0; j < 8; j++ )
 *                  image[ i ][ j ]   = func( i, j, count * 0.2 );
 *
 *          matrix.set_data( image );
 *
 *          count++;
 *          wait( 0.05 );
 *      }
 *  }
 *
 *  float func( float x, float y, float t )
 *  {
 *  //#define     DISPLAY_OFFSET  3.5
 *  #define     DISPLAY_OFFSET  0
 *  #define     SIZE            0.3
 *
 *      float   s;
 *
 *      x   = (x - DISPLAY_OFFSET) * SIZE;
 *      y   = (y - DISPLAY_OFFSET) * SIZE;
 *
 *      s   = cos( powf( x * x + y * y, 0.5 ) - t );
 *      return ( powf( s, 4.0 ) );
 *  }
 *  @endcode
 */

class PCA9622_LED8x8
{
public:
    enum FrameRate { DEFAULT_FRAMERATE   = 50 };

    /** Create a PCA9622_LED8x8 instance for the LED matrix board
     *  which is connected to specified I2C pins with specified address
     *
     * @param sda I2C-bus SDA pin
     * @param scl I2C-bus SCL pin
     * @param slave_adr (option) I2C-bus address (default: 0xA0)
     * @param fr (option) frame rate (in "Hz". default: 100)
     */
    PCA9622_LED8x8( PinName sda, PinName scl, char slave_adr = 0xA0, float fr = DEFAULT_FRAMERATE );

    /** Create a PCA9622_LED8x8 instance for the LED matrix board
     *  which is connected to specified I2C pins with specified address
     *
     * @param i2c_obj I2C object (instance)
     * @param slave_adr (option) I2C-bus address (default: 0xA0)
     * @param fr (option) frame rate (in "Hz". default: 100)
     */
    PCA9622_LED8x8( I2C &i2c_obj, char slave_adr = 0xA0, float fr = DEFAULT_FRAMERATE );

    /** Destructor of PCA9622_LED8x8
     */
    ~PCA9622_LED8x8();

    /** Frame rate change
     *
     *  Changes frame rate (scan rate of the matrix)
     *
     * @param rate Frame rate in "Hz"
     */
    void    frame_rate( float rate );

    /** Start operation
     *
     *  The LED matrix operation need to be started with this function.
     *  Scan line drawing by Ticker interrupt will be started.
     */
    void    start( void );

    /** Stop operation
     *
     *  This stops the operation.
     *  Ticker interrupt will be stopped
     */
    void    stop( void );

    /** Set the image to library internal buffer
     *
     *  8x8 pixel image will be copied into the library internal buffer.
     *  User don't need to care the timing of this function call because
     *  the library is using "ping-pong" buffering.
     *  This function writes data into in-active side of the buffer.
     *  The image will not be appear immediatly but from next scan start.
     *
     *  Each elements of the array should be brightness information
     *  in range of 0.0 to 1.0.
     *
     *  The 2 dimensional 8x8 array will be handled as..
     *
     *  p[0][0] : top-left
     *  p[0][7] : top-right
     *  p[7][0] : bottm-left
     *  p[7][7] : bottom-right
     *
     *  @param p image stored 2-dimensional float array
     */
    void    set_data( float p[ 8 ][ 8 ] );

private:
    I2C     *i2c_p;
    I2C     &i2c;
    char    address;                //  I2C slave address
    float   framerate;              //  frame rate in "Hz"
    int     in_operation;           //  ture, if scan is on-going
    int     line_counter;           //  counting scan line
    int     frame_counter;          //  counting frame number
    int     buffer_switch_request;  //  flag to switch the ping-pong buffer
    int     outgoing_buffer;        //  flag to select the buffer
    Ticker  t;
    char    pattern[ 2 ][ 8 ][ 8 ]; //  having 2 sets of 8x8 array for pin-pong buffering

    void    draw_a_line( void );    //  ISR called by Ticker
    void    initialize( void );     //  initialize function
};

#endif  //  MBED_PCA9622_LED8x8_H