#include "mbed.h"
#include "WS2812.h"
#include "PixelArray.h"
#include "GoalCounter.h"

#define WS2812_BUF 10 
#define NUM_COLORS 6
#define NUM_LEDS_PER_COLOR 2
#define COLOR 0x2f2020 

//Initialize instances of Timers for goal counters
Timer timer1;
Timer timer2;

// Initialize instace of a pixel array for LED strip 1 and 2
PixelArray px1(WS2812_BUF);
PixelArray px2(WS2812_BUF);

// The given numbers are calculated for the K64F MCU
WS2812 ws1(D4, WS2812_BUF, 0, 5, 5, 0);
WS2812 ws2(D5, WS2812_BUF, 0, 5, 5, 0);

// For printing information to serial console 
Serial pc(USBTX, USBRX);

GoalCounter gc1(D2, &timer1);
GoalCounter gc2(D3, &timer2);

float balltime1 = 0;
float speed1 = 0;

float balltime2 = 0;
float speed2 = 0;

uint8_t score1 = 0;
uint8_t score2 = 0;

// This function sets 1 - 10 to ws2812 led strip
void set_score(uint8_t score, PixelArray * px, WS2812 * ws) {
    px->SetAll(0); // null pixel array 
    
    for (int i=0; i < score; i++) {
        px->Set(i, COLOR); // set pixels to correspondig score
    }
    
    px->SetAllI(0x0f); // set intensity
    ws->write(px->getBuf()); // write buffer to the led strip
}

void print_score(uint8_t score, float balltime, float speed) {
    pc.printf("Score : %d \n\r", score);
    pc.printf("Time of ball pass : %f seconds\n\r", balltime);
    pc.printf("Speed of ball (34 mm diameter) : %f kph\n\r", speed);
}

// get information about score, balltime and speed from the GoalCounter 
void get_info(uint8_t * score, float * balltime, float * speed, GoalCounter * gc) {
    *score = gc->get_score();
    *balltime = gc->get_balltime();
    *speed = gc->get_ballspeed();
}

// Play animation for the winner, use pixel array 1, target strip passed as pointer
void winner_animation(WS2812 * ws) {
     // This function is modified and taken from the Brian Daniels WS2812 example
     // set up the colours we want to draw with
    int colorbuf[NUM_COLORS] = {0x2f0000,0x2f2f00,0x002f00,0x002f2f,0x00002f,0x2f002f};

    // for each of the colours (j) write out 10 of them
    // the pixels are written at the colour*10, plus the colour position
    // all modulus 60 so it wraps around
    for (int i = 0; i < WS2812_BUF; i++) {
        px1.Set(i, colorbuf[(i / NUM_LEDS_PER_COLOR) % NUM_COLORS]);
    }

    // now all the colours are computed, add a fade effect using intensity scaling
    // compute and write the II value for each pixel
    for (int j=0; j<WS2812_BUF; j++) {
        // px.SetI(pixel position, II value)
        px1.SetI(j%WS2812_BUF, 0xf+(0xf*(j%NUM_LEDS_PER_COLOR)));
    }


    // Now the buffer is written, rotate it
    // by writing it out with an increasing offset
    // 
    for (int i=0; i < 1000 ; i++) {
        for (int z=WS2812_BUF; z >= 0 ; z--) {
            ws->write_offsets(px1.getBuf(),z,z,z);
            wait(0.075);
        }
    }

}

int main()
{
    // use global intensity scaling
    ws1.useII(WS2812::GLOBAL);
    ws2.useII(WS2812::GLOBAL); 
      
    pc.printf("Initialization of the game... \n\r");
    
    while(1) {
        // polling - checking if goal was detected
        if (gc1.is_goal == 1) {
            get_info(&score1, &balltime1, &speed1, &gc1);
            set_score(score1, &px1, &ws1);
            pc.printf("Side 1 score \n\r");
            print_score(score1, balltime1, speed1);
            gc1.is_goal = 0;
        }
        else if (gc2.is_goal == 1) {
            get_info(&score2, &balltime2, &speed2, &gc2);
            set_score(score2, &px2, &ws2);
            pc.printf("Side 2 score \n\r");
            print_score(score2, balltime2, speed2);
            gc2.is_goal = 0;     
        }
        // when game is over, print game replay and winning animation
        if (score1 >= 10 || score2 >= 10) {
            pc.printf("Game over \n\r");
            pc.printf(((score1 > score2) ? "Team 1 is winner" : "Team 2 is winner"));
            
            pc.printf("\n Game score Team 1 : \n\r");
            
            for (int i=1; i<= score1; i++) {
                    balltime1 = gc1.get_balltime(i);
                    speed1 = gc1.get_ballspeed(i);
                    print_score(i, balltime1, speed1);
                }
            
            pc.printf("Game score Team 2 : \n\r");
            
            for (int i=1; i<= score2; i++) {
                    balltime2 = gc2.get_balltime(i);
                    speed2 = gc2.get_ballspeed(i);
                    print_score(i, balltime2, speed2);
                }
            
            // play winner animation on winning team's LED strip
            winner_animation(((score1 > score2) ? &ws1 : &ws2));
              
            return 0;
        }  

    }
}
