#ifndef GOALCOUNTER_H
#define GOALCOUNTER_H

#include "mbed.h"

/** GoalCounter class.
 *  Simple implementation of goal counter for table football using laser diode
 *  and phototransistor. Output of the phototransistor is connected to the digital
 *  input pin of MCU. When the laser beam is interrupted by ball passing the the cage
 *  Timer starts counting and measuring width of the pulse and if the duration of pulse
 *  is not longer than two seconds, goal is detected and _score variable is incremented.
 *   
 *  Pulse duration is also used for the calculation of ball passing speed. However,
 *  measurement is not precise.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "GoalCounter.h"
 * 
 * Timer t;
 * GoalCounter gc(D2, &t);
 * 
 * int main() {
 *  while(1) {
 *       if (gc.is_goal)
 *           goal(); // call some function
 *       }
 *   
 *   }
 * 
 * @endcode
 * 
 **/
 
class GoalCounter {
public:
    /**
    *   Constructor
    *
    *   @param pin Input pin where is the outpout of phototransistor connected
    *   @param timer reference to the Timer instance
    *
    */
    GoalCounter(PinName pin, Timer * t);
    /**
    *   Destructor
    *
    *   Destroys instance
    */
    ~GoalCounter();   
    /**
    *   Reads value of the timer, 
    *   this function is called from the interrupt pin fall edge callback
    */
    void tstart();
    /**
    *   Reads value from the timer and measures interval between end and beginning
    *   Called from the interrupt pin rise edge callback
    */
    void tstop();
    /**
    *   Returns score
    *   @return score score of the game
    */
    uint8_t get_score();
    /**
    *   Returns time of a ball pass for a given score
    *    @param score of the game
    *    @return time of a ball pass in seconds
    */
    float get_balltime(uint8_t score);
    /**
    *   Returns time of a ball pass for the current score
    *    @return time of a ball pass in seconds
    */
    float get_balltime();
    /**
    *   Returns speed of a ball pass for a given score
    *    @param score of the game
    *    @return speed of a ball pass in kph
    */
    float get_ballspeed(uint8_t score);
    /**
    *   Returns speed of a ball pass for the current score
    *    @return speed of a ball pass in kph
    */
    float get_ballspeed();
    /**
    *  Variable is 1 (true) when the goal is detected
    */
    volatile uint8_t is_goal;

private:
    InterruptIn * _interrupt;
    volatile uint8_t _score;
    Timer * _t;
    float _balltimes[11]; // sorry for the FP arithmetics, i was too lazy to rewrite it
    float _time;
    float _begin;
    float _end;
    float _last_end;
    float _time_between;      
};

#endif 