#ifndef SABERTOOTH2X25_H
#define SABERTOOTH2X25_H

#include"mbed.h"

//command
#define DRIVE_FORWARD_MOTOR_ONE  0
#define DRIVE_BACKWARD_MOTOR_ONE  1
#define MIN_VOLTAGE  2
#define MAX_VOLTAGE  3
#define DRIVE_FORWARD_MOTOR_TWO  4
#define DRIVE_BACKWARD_MOTOR_TWO  5
#define DRIVE_MOTOR_ONE_7BIT  6
#define DRIVE_MOTOR_TWO_7BIT  7
//mixed mode command
#define DRIVE_FORWARD_MIXED_MODE 8
#define DRIVE_BACKWARD_MIXED_MODE 9
#define TURN_RIGHT_MIXED_MODE 10
#define DRIVE_TURN_LEFT_MIXED_MODE 11
#define DRIVE_FORWARD_BACK_7BIT  12
#define TURN_7BIT  13

/** Sabertooth2x25 class
 *
 * This class is a class for controlling a motor by Sabertooth2x25(Packetized Serial Mode). 
 */

class Sabertooth2x25 {
public:
    Sabertooth2x25(PinName tx, int baudrate);

    /** This is used to command motor 1 to drive forward.
     * This is used to command motor 1 to drive forward. Valid data is 0-127 for off to full forward drive.
     * If a command of 0 is given, the Sabertooth will go into power save mode for motor 1 after approximately 4 seconds.
     *
     * @param device_address The set device address 
     * @param speed motor spped(Valid data is 0-127)
     */
    void forward_motor_1(int device_address, int speed);

    /**This is used to command motor 1  to drive backward.
     * This is used to command motor 1  to drive backward. Valid data is 0-127 for off to full reverse drive.
     * If a command of 0 is given, Sabertooth will go into power save mode for motor 1 after approximately 4 seconds.
     *
     * @param device_address The set device address 
     * @param speed motor spped(Valid data is 0-127)
     */
    void backward_motor_1(int device_address, int speed);

    /** This is used to set a custom minimum voltage for the battery feeding the Sabertooth.
     * This is used to set a custom minimum voltage for the battery feeding the Sabertooth.  If the battery voltage drops below this value, the output will shut down.
     * This value is cleared at startup, so much be set each run. The value is sent in .2 volt increments with a command of zero corresponding to 6v, which is the minimum.
     * Valid data is from 0 to 120. The function for converting volts to command data is
     * Value = (desired volts-6) x 5
     *
     * @param device_address The set device address
     * @param data Value = (desired volts-6) x 5 (Valid data is from 0 to 120)
     */
    void min_voltage(int device_address, int data);

    /** This is used to set a custom maximum voltage.
     * This is used to set a custom maximum voltage. If you are using a power supply that cannot sink current such as an ATX supply,
     * the input voltage will rise when the driver is regenerating (slowing down the motor) Many ATX type supplies will shut down if the output voltage on the 12v supply rises beyond 16v.
     * If the driver detects an input voltage above the set limit, it will put the motor into a hard brake until the voltage drops below the set point again.
     * This is inefficient, because the energy is heating the motor instead of recharging a battery, but may be necessary.
     * The driver comes preset for a maximum voltage of 30V.  The range for a custom maximum voltage is 0v-25v. The formula for setting a custom maximum voltage is
     * Value = Desired Volts*5.12
     * If you are using any sort of battery, then this is not a problem and the max voltage should be left at the startup default.
     *
     * @param device_address The set device address
     * @param data Value = Desired Volts*5.12 (0-25v)
     */
    void max_voltage(int device_address, int data);

    /** This is used to command motor 2 to drive forward.
     * This is used to command motor 2 to drive forward. Valid data is 0-127 for off to full forward drive.
     * If a command of 0 is given, the Sabertooth will go into power save mode for motor 2 after approximately 4 seconds.
     *
     * @param device_address The set device address
     * @param spped motor speed(Valid data is 0-127)
     */
    void forward_motor_2(int device_address, int speed);

    /** This is used to command motor 2 to drive backward.
     * This is used to command motor 2 to drive backward. Valid data is 0-127 for off to full reverse drive.
     * If a command of 0 is given, the Sabertooth will go into power save mode after approximately 4 seconds.
     *
     * @param device_address The set device address
     * @param data Value = Desired Volts*5.12 (0-25v
     */
    void backward_motor_2(int device_address, int speed);

    /** This command is used to drive motor 1.
     * This command is used to drive motor 1. Instead of the standard commands 0 and 1, this one command can be used to drive motor 1 forward or in reverse,
     * at a cost of lower resolution. A command of 0 will correspond to full reverse, and a command of 127 will command the motor to drive full forward. A command of 64 will stop the motor.
     *
     * @param device_address The set device address
     * @param data  A command of 0 will correspond to full reverse, and a command of 127 will command the motor to drive full forward.
     */
    void drive_motor_1(int device_address, int speed);

    /** This command is used to drive motor 2.
     * This command is used to drive motor 2. Instead of the standard commands 4 and 5, this one command can be used to drive motor 1 forward or in reverse,
     * at a cost of lower resolution. A command of 0 will correspond to full reverse, and a command of 127 will command the motor to drive full forward. A command of 64 will stop the motor.
     *
     * @param device_address The set device address
     * @param data  A command of 0 will correspond to full reverse, and a command of 127 will command the motor to drive full forward.
     */
    void drive_motor_2(int device_address, int speed);

    /** This is used to command the vehicle to drive forward in mixed mode.
     * This is used to command the vehicle to drive forward in mixed mode. Valid data is 0-127 for off to full forward drive.
     *
     * @param device_address The set device address
     * @param speed Valid data is 0-127 for off to full forward drive.
     */
    void drive_forward_mixed_mode(int device_address, int speed);

    /** This is used to command the vehicle to drive backwards in mixed mode. 
     * This is used to command the vehicle to drive backwards in mixed mode. Valid data is 0-127 for off to full reverse drive.
     *
     * @param device_address The set device address
     * @param Valid data is 0-127 for off to full reverse drive.
     */
    void drive_backward_mixed_mode(int device_address, int speed);

    /** This is used to command the vehicle to turn right in mixed mode.
     * This is used to command the vehicle to turn right in mixed mode. Valid data is 0-127 for zero to maximum turning speed.
     * 
     * @param device_address The set device address
     * @param speed Valid data is 0-127 for zero to maximum turning speed.
     */
    void turn_right_mixed_mode(int device_address, int speed);

    /** This is used to command the vehicle to turn leftt in mixed mode.
     * This is used to command the vehicle to turn leftt in mixed mode. Valid data is 0-127 for zero to maximum turning speed.
     *
     * @param device_address The set device address
     * @param speed Valid data is 0-127 for zero to maximum turning speed.
     */
    void drive_turn_left_mixed_mode(int device_address, int speed);

    /** This is used to command the vehicle to move forwards or backwards.
     * This is used to command the vehicle to move forwards or backwards. A command of 0 will cause maximum reverse, 64 will cause the vehicle to stop, and 127 will command full forward.
     *
     * @param device_address The set device address
     * @param speed A command of 0 will cause maximum reverse, 64 will cause the vehicle to stop, and 127 will command full forward.
     */
    void drive_forwards_back(int device_address, int speed);

    /** This is used to command the vehicle turn right or left.
     * This is used to command the vehicle turn right or left. A command of 0 will cause maximum left turn rate,
     * 64 will cause the vehicle to stop turning , and 127 will command maximum right turn rate.
     *
     * @param device_address The set device address
     * @param spped . A command of 0 will cause maximum left turn rate,64 will cause the vehicle to stop turning , and 127 will command maximum right turn rate.
     */
    void turn(int device_address, int speed);


private:
    void sendCommand(int address, int command, int data);
    Serial device_;
    unsigned char checksum_;
};


#endif