/**
  ******************************************************************************
  * @file    RMS_UART.cpp
  * @author  Surasak N
  * @version V2
  * @date    20/09/2016
  * @brief   Command for Mini-RMS <> RMS via UART
  ******************************************************************************/
  
#include "mbed.h"
#include "main.h"

#ifdef EXT_SERIAL
    /* Using UART1 as in Debuger (ST-Link) */
    Serial serial_device(SERIAL_TX, SERIAL_RX);
#else
    /* Using UART2 via external UART to USB */
    Serial serial_device(PA_3, PB_3);
#endif

char Msg_RxBuf[MSG_BUF_SIZE + 1];               // Command Buffer 
int  Msg_index;                                 // Command index

char Content_RxBuf[CONTENT_BUF_SIZE + 1];       // Contents Buffer 
int  Content_index;                             // Contents index

char FileName[MSG2FILENAME];                    // Command Buffer 
FileRequest_CMD_Type reqFileCmdType = REQ_NONE; // An Events Request and management
bool MsgContentManagement = false;              // Command and Contents buffer management
bool SystemStandBy = false;                     // Mini-RMS system standby flag
bool OverBufSize   = false;                     // Content buffer size overload

/**
 * @brief   Initials bsp and interrupt of serial communication
 * @note    BPS have 9600bps only 
 * @retval 
 */
void Init_SerialDevice()
{
    /* Initial begining bps with 9600bps */
    serial_device.baud(BPS_9600);
    
    /* Initial interrupt callback function */
    serial_device.attach(&RxMsgInterruptCallback);
}

/**
 * @brief   Intertupt callback function
 * @note    Management fill buffer direction
 * @retval 
 */
void RxMsgInterruptCallback()
{
    /* Received serial Rx interrupt */
    
    /* Check buffer direction */
    if(MsgContentManagement == false)
    {
        /* Fill buffer to commands */
        DoTheMsgCmd();
    }
    else
    {
        /* Fill buffer to contents */
        DoTheContent();
    }
}

/**
 * @brief   Fill serial data as in contents buffer
 * @note    
 * @retval
 */
void DoTheContent()
{   
    // Note: you need to actually read from the serial to clear the RX interrupt
  
    /* Fill data as serial to contents buffer */
    Content_RxBuf[Content_index] = serial_device.getc();
        
    /* Shift contents index */
    Content_index++;
    
    /* Check end of content */
    CheckContentMsg();

}

 /**
 * @brief   Fill serial data as in command buffer
 * @note    Command type and FileName
 * @retval
 */
void DoTheMsgCmd()
{
        
    // Note: you need to actually read from the serial to clear the RX interrupt
    /* Fill data as serial to command and file name buffer */
    Msg_RxBuf[Msg_index] = serial_device.getc();
    
    /* Shift command and filename index */
    Msg_index++;
    
    /* Check command type */
    CheckCmdType();
}

/**
 * @brief   Checking and processing contents data
 * @note    Check command *end* and clear buffer
 * @retval
 */
void CheckContentMsg()
{
    
    /* Found the *end* */
    if(strstr(Content_RxBuf,CMD_END_OF_CONTENT))
    {
        /* Call function for check contents */
        SendContentToFile();
    }
    
    /* Check content over buffer size */
    if(Content_index == CONTENT_BUF_SIZE)
    {
        /* Clear content buffer and index */
        ClearContentIndexAndBuf();
        
        /* Set content buffer overload flag */
        OverBufSize = true;
        
    }
}

 /**
 * @brief   Checking and processing command type
 * @note    $ls, $ed, $rd, $rs and $df and etc.
 * @retval
 */
void CheckCmdType()
{   
    // Note : Procese the command type and file name
    
    /* Command type is list file ($ls) */
    if(strstr(Msg_RxBuf,CMD_LISTFILE))
    {        
        /* Set REQ_LISTDIR  event's reqFileCmdType */
        reqFileCmdType = REQ_FLAG_OK;
        
        /* Set flag evetnt */
        /* Waiting! main program call event */
        UART_File_event = true;
        
    }
    /* RMS sent "LK" acknowledge */
    else if(strstr(Msg_RxBuf,RMS_STATUS_LIST_OK))
    {
        /* Set REQ_LISTDIR  event's reqFileCmdType */
        reqFileCmdType = REQ_LISTDIR;
        
        /* Set flag evetnt */
        /* Waiting! main program call event */
        UART_File_event = true;
        
        /* Clear command index and buffer */
        ClearCmdIndexAndBuf();
        
    }
    /* RMS sent "RK" acknowledge */
    else if(strstr(Msg_RxBuf,RMS_STATUS_READ_OK))
    {
        /* Categorization command type */
        CheckReadEditCmd();
    }
    /* Got the *.csv file name */
    else if(strstr(Msg_RxBuf,FOUND_CSV_FILE))
    {
        /* Categorization ommand type */
        CheckReadEditCmd();
    }
    /* Got the *.xml file name */  
    else if(strstr(Msg_RxBuf,FOUND_XML_FILE))
    {
        /* Categorization command type */
        CheckReadEditCmd();
    }
    /* Got the *.log file name */
    else if(strstr(Msg_RxBuf,FOUND_LOG_FILE))
    {
        /* Categorization command type */
        CheckReadEditCmd();
    }
    /* Command type is system restart ($rs) */
    else if(strstr(Msg_RxBuf,CMD_SYS_RESTART))
    {
        /* Set SystemStandBy to not active */
        SystemStandBy = false;
        
        /* Request system restart */
        NVIC_SystemReset();
    }
    /* Command type is help ($-h) */
    else if(strstr(Msg_RxBuf,CMD_HELP))
    {
        /* List all command */
        GetHelpCmd();
    }
    /* Command type is stop ($sp) */
    else if(strstr(Msg_RxBuf,CMD_SYS_PAUSE))
    {
        /* Set SystemStandBy to active */
        SystemStandBy = true;
    }
    /* Command may be mismatched type */
    else
    {
        /* Not receive true type command */
        if(Msg_index >= MSG2CMD)
        {
            if(!((strstr(Msg_RxBuf,RMS_STATUS_LIST_OK))||
                 (strstr(Msg_RxBuf,RMS_STATUS_READ_OK))||
                 (strstr(Msg_RxBuf,CMD_LISTFILE))      || 
                 (strstr(Msg_RxBuf,CMD_READFILE))      || 
                 (strstr(Msg_RxBuf,CMD_WRITEFILE)      ||
                 (strstr(Msg_RxBuf,CMD_HELP))          || 
                 (strstr(Msg_RxBuf,CMD_SYS_RESTART))   ||
                 (strstr(Msg_RxBuf,CMD_SYS_PAUSE))     || 
                 (strstr(Msg_RxBuf,CMD_DELETEFILE)))))
            {
                /* Call command error function */ 
                GetCmdError();
            }
        }
    }

    /* Command and file name buffer are full */
    if(Msg_index == MSG_BUF_SIZE)
    {
       /* Clear command index and buffer */
       ClearCmdIndexAndBuf();
    }
}

/**
 * @brief   Categorization commands type
 * @note    Received file name and type
 * @retval
 */
void CheckReadEditCmd()
{
    /* Command type is read file ($rd) */
    if(strstr(Msg_RxBuf,CMD_READFILE))
    {   
    
        /* Check file name */
        GetFileName();

        /* Set REQ_READ  event's reqFileCmdType */
        reqFileCmdType = REQ_READ_CHECK;
        
        /* Set flag evetnt */
        /* Waiting! main program call event */
        UART_File_event = true;
    }
    else if(strstr(Msg_RxBuf,RMS_STATUS_READ_OK))
    {        
        /* Set REQ_READ  event's reqFileCmdType */
        reqFileCmdType = REQ_READ;
        
        /* Set flag evetnt */
        /* Waiting! main program call event */
        UART_File_event = true;
    }
    
    /* Command type is write/create file ($ed) */
    else if(strstr(Msg_RxBuf,CMD_WRITEFILE))
    {
        /* Set REQ_READ  event's reqFileCmdType */
        reqFileCmdType = REQ_FLAG_OK;
        
        /* Set flag evetnt */
        /* Waiting! main program call event */
        UART_File_event = true;
        
        /* Check file name */
        GetFileName();
        
        /* Set event flag for fill contents buffer */
        MsgContentManagement = true; 

    }
    /* Command type is delete ($df) */
    else if(strstr(Msg_RxBuf,CMD_DELETEFILE))
    {
        /* Check file name */
        GetFileName();
        
        /* Set REQ_DELETE  event's reqFileCmdType */
        reqFileCmdType = REQ_DELETE;
        
        /* Set flag evetnt */
        /* Waiting! main program call event */
        UART_File_event = true;
    }
    else
    {
        /* Call command error function */ 
        GetCmdError();
    }
}

/**
 * @brief   Received command error 
 * @note    $??
 * @retval
 */
void GetCmdError()
{   
    /* Send error status "ER" */
    serial_device.printf(EC_CMD_ERROR);
    
    /* Clear command index and buffer */
    ClearCmdIndexAndBuf();
}

/**
 * @brief   FileName process
 * @note    Fill file name as buffer
 * @retval
 */
void GetFileName()
{   
    /* Copy file name from Msg to FileName */
    memcpy(FileName,&Msg_RxBuf[3],Msg_index - 3);
    
    /* Clear command index and buffer */
    ClearCmdIndexAndBuf();
}

/**
 * @brief   Request sent contents to main 
 * @note    
 * @retval
 */
void SendContentToFile()
{   
    /* Set fill buffer to command */
    MsgContentManagement = true;
    
    /* Set REQ_WRITE  event's reqFileCmdType */
    reqFileCmdType = REQ_WRITE;
    
    /* Set flag evetnt */
    /* Waiting! main program call event */
    UART_File_event = true;
    
}

/**
 * @brief   Command help
 * @note    $-h
 * @retval
 */
void GetHelpCmd()
{   
    /* List all commands*/
    serial_device.printf("\r\n$ls : List fils as in directory");
    serial_device.printf("\r\n$rd [File Name] : Read fils as file name");
    serial_device.printf("\r\n$df [File Name] : Delete fils as file name");
    serial_device.printf("\r\n$ed [File Name] [Content] : Write/Create fils as in directory with content");
    serial_device.printf("\r\n\tNote : Content must contain EOF signature (*end*) to indicate end of file");
    serial_device.printf("\r\n\te.g. ed TestFile.xml test content *end*");
    serial_device.printf("\r\n$rs : Mini-RMS system restart");
    serial_device.printf("\r\n$sp : Mini-RMS system stop");
    serial_device.printf("\r\n\tNote : Need $rs for resumming Mini-RMS system");
    
    /* Clear buffer */
    ClearCmdIndexAndBuf();
}
/**
 * @brief   Clear index and buffer of command and file name
 * @note    
 * @retval
 */
void ClearCmdIndexAndBuf()
{   
    /* Clear counter */
    Msg_index = 0;
    
    /* Clear buffer */
    memset(Msg_RxBuf,_SPACE,MSG_BUF_SIZE);
}

/**
 * @brief   Clear index and buffer of contents 
 * @note 
 * @retval
 */
void ClearContentIndexAndBuf()
{   
    /* Clear counter */
    Content_index = 0;
    
    /* Clear buffer */
    memset(Content_RxBuf,_SPACE,CONTENT_BUF_SIZE);
}

/**
 * @brief   Clear fileName buffer
 * @note 
 * @retval
 */
void ClearFileNameBuf()
{   
    /* Clear buffer */
    memset(FileName,_SPACE,MSG2FILENAME);
}
////////////////////////////////////////////////* END *////////////////////////////////////////////////