/**
  ******************************************************************************
  * @file    GPSGms6.h
  * @author  Narut T
  * @version V1
  * @date    19/05/2016
  * @brief   Header for Library GPS GMS6
  ******************************************************************************/

/* ###############  Constant Defination  ################## */
#ifndef GPSGMS6_H
#define GPSGMS6_H

#define HEADER_SIZE                 5           /** Header Size*/
#define GPRMC_TIME_SIZE             10          /** GPRMC Time Size */
#define GPRMC_STATUS_SIZE           1           /** GPRMC Status Size */
#define GPRMC_LATITUDE_SIZE         9           /** GPRMC Latitude Size */  
#define GPRMC_LATITUDE_HEMI_SIZE    1           /** GPRMC Latitude hemi Size */
#define GPRMC_LONGITUDE_SIZE        10          /** GPRMC Longitude Size */
#define GPRMC_LONGITUDE_HEMI_SIZE   1           /** GPRMC Longitude hemi Size */
#define GPRMC_SPEED_SIZE            5           /** GPRMC speed Size */
#define GPRMC_COURSE_SIZE           5           /** GPRMC Course Size */
#define GPRMC_DATE_SIZE             6           /** GPRMC Date Size */
#define GPRMC_MAGNETIC_SIZE         5           /** GPRMC Magnetic Size */
#define GPRMC_MAGNETIC_DIR_SIZE     1           /** GPRMC Magnetic Dir Size */
#define GPRMC_INDICATOR_SIZE        1           /** GPRMC Indicator Size */

#define INVALID_VALUE               0xFFFF      /** Invalid value */
#define RX_BUF_SIZE                 100         /** Receive Buffer Size */

#define GPS_BAUD_RATE               9600        /** GPS Baudrate*/
#define GPS_STRING_BUFFER_SIZE      3           /** String conversion buffer size */

#define GPS_MESSAGE_HEADER_PREFIX   '$'         /** Every format in GPS have this char for prefix of header */
#define GPS_DELIMITER               ','         /** Delimiter between section */
#define GPS_DEFAULT_DATA_VALUE      ' '         /** Default data value */
#define GPS_TIME_SECOND_OFFSET      4           /** Second Offset in GPS Time */
#define GPS_TIME_SECOND_SIZE        2           /** Second Size in GPS Time  */    
#define GPS_TIME_MINUTE_OFFSET      2           /** Minute Offset in GPS Time  */
#define GPS_TIME_MINUTE_SIZE        2           /** Minute Size in GPS Time  */
#define GPS_TIME_HOUR_OFFSET        0           /** Hour Offset in GPS Time  */
#define GPS_TIME_HOUR_SIZE          2           /** Hour Size in GPS Time  */
#define GPS_TIME_DATE_OFFSET        0           /** Date Offset in GPS Date  */
#define GPS_TIME_DATE_SIZE          2           /** Date Size in GPS Date  */
#define GPS_TIME_MONTH_OFFSET       2           /** Month Offset in GPS Date  */
#define GPS_TIME_MONTH_SIZE         2           /** Month Size in GPS Date  */
#define GPS_TIME_YEAR_OFFSET        4           /** Year Offset in GPS Date  */
#define GPS_TIME_YEAR_SIZE          2           /** Year Size in GPS Date  */

#define INVALID_TIME_DATE           0           /** Invalid date value  */
#define INVALID_TIME_MONTH          0           /** Invalid month value   */
#define INVALID_TIME_YEAR           0           /** Invalid year value   */

/* ###############  Method Defination  ################## */
/** @brief  Check if the indicator is valid
  * @param  indicator - a char that represent the indicator message
  * @retval True - if indicator is valid , otherwise, false .
  */
#define IS_INDICATOR_VALID(indicator)  ((indicator == (char)'A') || (indicator == (char)'D')|| (indicator == (char)'E'))

/** @brief  Check header is GPRMC header 
  * @param  pointer to char array that contain message
  * @retval True - if header is GPRMC , otherwise, false.
  */
#define IS_HEADER_GPRPC(buf)        (buf[index] == 'G' &&    \
                                     buf[index+1] == 'P' &&  \
                                     buf[index+2] == 'R' &&  \
                                     buf[index+3] == 'M' &&  \
                                     buf[index+4] == 'C'  )   
                                       

/* ###############  Enum  ################## */
typedef enum GPS_ProcessState {
    GPS_Process_Start = 0,          /** State Start processing   */
    GPS_Process_Header,             /** State process header   */
    GPS_Process_Time,               /** State process Time   */
    GPS_Process_Status,             /** State process status   */
    GPS_Process_Latitude,           /** State process Latitude   */
    GPS_Process_Latitude_hemis,     /** State process Latitude hemis   */
    GPS_Process_Longitude,          /** State process Longitude   */
    GPS_Process_Longitude_hemis,    /** State process Longitude hemis  */   
    GPS_Process_Speed,              /** State process Speed   */
    GPS_Process_Course,             /** State process Course   */
    GPS_Process_Date,               /** State process Date   */
    GPS_Process_Magnetic,           /** State process Magnetic   */
    GPS_Process_Magnetic_Dir,       /** State process Magnetic Dir   */
    GPS_Process_Indicator,          /** State process Indicator   */
    GPS_Process_Complete ,          /** State process Complete   */
    GPS_Process_SIZE                /** NUMBER OF STATEs */
} GPS_ProcessState;

typedef enum GPS_ProcessStatus {
    GPS_Status_Valid,               /** Status valid   */
    GPS_Status_Empty,               /** Status data is empty  */
    GPS_Status_NotEnough,           /** Status Receive Buffer is not enough to process  */
} GPS_ProcessStatus;


/* ###############  Structure  ################## */
typedef struct {
    char header[HEADER_SIZE+1];                         /** Specific section for header data  */
    char time[GPRMC_TIME_SIZE+1];                       /** Specific section for time data  */
    char status[GPRMC_STATUS_SIZE+1];                   /** Specific section for status data  */
    char latitude[GPRMC_LATITUDE_SIZE+1];               /** Specific section for latitude data  */    
    char latitude_hemi[GPRMC_LATITUDE_HEMI_SIZE+1];     /** Specific section for latitude hemi data  */    
    char longitude[GPRMC_LONGITUDE_SIZE+1];             /** Specific section for longitude data  */    
    char longitude_hemi[GPRMC_LATITUDE_HEMI_SIZE+1];    /** Specific section for longitude hemi data  */    
    char speed[GPRMC_SPEED_SIZE+1];                     /** Specific section for speed data  */    
    char course[GPRMC_COURSE_SIZE+1];                   /** Specific section for Course data  */    
    char date[GPRMC_DATE_SIZE+1];                       /** Specific section for Date data  */   
    char magnetic[GPRMC_MAGNETIC_SIZE+1];               /** Specific section for Magnetic data  */   
    char magnetic_dir[GPRMC_MAGNETIC_DIR_SIZE+1];       /** Specific section for Magnetic dir data  */   
    char indicator[GPRMC_INDICATOR_SIZE+1];             /** Specific section for Indicator data  */       
} GPRMC_Data_TypeDef;


typedef struct {
    GPS_ProcessState state;                             /** indicate index of the table */   
    int              size;                              /** section size */   
    char *           p_val;                             /** pointer to variable to be store */   
} GPRMC_Tbl_TypeDef;


/* ###############  Class  ################## */
class GPSGms6
{
public:

    /**
     * Constructor
     */
    GPSGms6();

    /**
     * Start GPS 
     */
    void start_GPS();
    
    /**
     * Get UTC Time 
     * @returns
     *   UTC Time
     */
    tm UTCTime();
    /** Get Latest GPRMC Data
     *
     * @returns
     *   Latest GPRMC Data
     */
    GPRMC_Data_TypeDef * latestGPRMC();

    /**  Get Valid Data Interval
     *
     * @returns
     * Valid Data Interval
     */
    GPRMC_Data_TypeDef * validGPRMC();

    /**  Get availability of gprmc
     *
     * @returns
     * 'true' - if data is available
     */
    bool available();


private:

};

#endif