/**
  ******************************************************************************
  * @file    FileManager.h
  * @author  Narut T
  * @version V1
  * @date    19/05/2016
  * @brief   Header file for File Manager
  ******************************************************************************/
/* ###############  Constant Defination  ################## */

#ifndef FILEMANAGER_H
#define FILEMANAGER_H

#define DEVICE_NAME             "sd"                        /*!< Device directory */                       

// DEVICE_NAME must be put in between "/" for file name 
#define PREFIX_FILE_NAME         "/sd/"                     /*!< Prefix File Name */
#define SETUP_FILE_NAME         "/sd/RMS_Tester.xml"        /*!< Setup File Name */
#define GPS_LOG_FILE_NAME       "/sd/YYYY-MM-DD.gps.csv "   /*!< GPS Log File Name */
#define RTL_LOG_FILE_NAME       "/sd/YYYY-MM-DD.rtl.csv"    /*!< RMS Log File Name */
#define VARIABLE_FILE_NAME      "/sd/20160216185627_upload.xml" /*!< Mini RMS Log File Name */
#define MINIRMS_LOG_FILE_NAME   "/sd/miniRMS.log"

/** XML Tag */
#define GPS_TAG                 "<Gps>"                     /*!< GPS Tag */
#define DATA_TAG                "<Data>"                    /*!< Data Tag */
#define VAR_LIST_FILE_TAG       "<List_File_Name>"          /*!< Variable List File Name Tag */
#define RMS_DEVICE_TAG          "<Device>"                  /*!< RMS Device Tag */
#define RMS_KEY_TAG             "<Key>"                     /*!< RMS Key Tag*/
#define UPDATE_INTERVAL_TAG     "<Update_Interval>"         /*!< Update Internal Tag */
#define VAR_NAME_TAG            "<varName>"                 /*!< Variable Name Tag */
#define VAR_ADDR_TAG            "<varAddress>"              /*!< Variable Address Tag */
#define VAR_TYPE_TAG            "<varType>"                 /*!< Variable Type Tag */
#define VAR_LSB1_TAG            "<LSB1>"                    /*!< Variable LSB1 Tag */
#define VAR_LSB2_TAG            "<LSB2>"                    /*!< Variable LSB2 Tag */
#define VAR_BITMASK_TAG         "<BitMask>"                 /*!< Variable Bit Mask Tag */
#define VAR_UNIT_TAG            "<Unit>"                    /*!< Variable Unit Tag */

/** XML Text Size */
#define XMLTEXT_SIZE            20                          /*!< XML Text Maximum size */
#define VAR_NAME_MAX_SIZE       20                          /*!< Variable Name Maximum Size  */
#define VAR_ADDR_MAX_SIZE       10                          /*!< Variable Address Maximum Size  */
#define VAR_TYPE_MAX_SIZE       3                           /*!< Variable Type Maximum Size  */
#define VAR_LSB1_MAX_SIZE       3                           /*!< Variable LSB1 Maximum Size  */
#define VAR_LSB2_MAX_SIZE       10                          /*!< Variable LSB2 Maximum Size  */
#define VAR_BITMASK_MAX_SIZE    5                           /*!< Variable Bit Mask  Maximum Size  */
#define VAR_UNIT_MAX_SIZE       20                          /*!< Variable Unit Maximum Size  */

#define MAX_FILE_NAME_SIZE      40                          /*!< Maximum File name size to be supported  */
#define MAX_VAR                 50                          /*!< Maximum variable to be supported  */

#define RMS_HEADER_TIME         "Time"                      /*!< RMS Header Time tag  */
#define RMS_HEADER_MSECOND      "MSecond"                   /*!< RMS MSecond Time tag  */
#define READ_FILE_BUFFER_SIZE   1024                        /*!< Read file buffer size */
#define RENAME_FILE_BUFFER_SIZE 5                           /*!< Rename File buffer size */

/** Offset for /sd/YYYY-MM-DD-  string */
#define TIMESTAMP_YEAR_OFFSET   4                           /*!< Offset for year  */
#define TIMESTAMP_YEAR_SIZE     4                           /*!< Size for year  */
#define TIMESTAMP_MONTH_OFFSET  9                           /*!< Offset for month  */
#define TIMESTAMP_MONTH_SIZE    2                           /*!< Size for month  */
#define TIMESTAMP_DATE_OFFSET   12                          /*!< Offset for date  */
#define TIMESTAMP_DATE_SIZE     2                           /*!< Size for date  */
#define YEAR_4DIGITS_OFFSET     1900                        /*!< Offset to convert 2 digits year to 4 digits */

extern bool m_sdCardIsPresent;
/* ###############  Enum  ################## */

/* Exported types ------------------------------------------------------------*/
/** @defgroup ReadingFileState Exported Types
  * @{
  */

/** 
  * @brief  Reading File State structures definition  
  */ 
typedef enum  {
    STATE_FINDING,                      /*!< Finding */
    STATE_FOUND_DATA,                   /*!< Found Data tag */
    STATE_FOUND_DATA_INTERVAL,          /*!<< Found update internal of tag*/
    STATE_FOUND_GPS,                    /*!< Found GPS tag */
    STATE_FOUND_GPS_INTERVAL,           /*!< Found update internal of GPS*/
} ReadingFileState;

/** 
  * @brief  Write File Status structures definition  
  */ 
typedef enum  {
    FILE_ERROR,
    FILE_OK
} WritingFileStatus;


/* ###############  Structure   ################## */


/** 
  * @brief   Structure of Variable List File
  */ 
typedef struct {
    char varName[VAR_NAME_MAX_SIZE];                /*!< char array Variable Name */
    char varAddress[VAR_ADDR_MAX_SIZE+1];           /*!< char array Variable Address */
    char varType[VAR_TYPE_MAX_SIZE];                /*!< char array Variable Type */
    char varLSB1[VAR_LSB1_MAX_SIZE];                /*!< char array Variable LSB1 */
    char varLSB2[VAR_LSB2_MAX_SIZE];                /*!< char array Variable LSB2 */
    char varBitMask[VAR_BITMASK_MAX_SIZE];          /*!< char array Variable Bit Mask */
    char varUnit[VAR_UNIT_MAX_SIZE];                /*!< char array Variable Unit */
} Variable_Data_TypeDef;



/* ###############  Function Prototype  ################## */
void FILEMANAGER_ReadSetupFile();
void FILEMANAGER_DeleteFile(char filename[]);
int  FILEMANAGER_GPSInterval();
int  FILEMANAGER_DataInterval();
void FILEMANAGER_LogGPSData(time_t timestamp ,char lat[], char longti[]);
void FILEMANAGER_LogSystemData(float gps_interval,float rms_interval,bool isResetCausedByWdg = false);
Variable_Data_TypeDef * FILEMANAGER_ReadVarFile();
int FILEMANAGER_GetAmountVarList();
Variable_Data_TypeDef * FILEMANAGER_GetVarList();
void FILEMANAGER_LogRMSData(time_t timestamp ,float * var, int size, bool *  p_headerRequired, uint32_t msec);
void FILEMANAGER_LogRMSHeader(FILE *fp, time_t timestamp, char file_name[]);
bool FILEMANAGER_IsFileExist(char filename[]);
void FILEMANAGER_listDir(Serial* serial);
void FILEMANAGER_DisplayFile(Serial* serial, char* filename);
void FILEMANAGER_AddPrefixFilename(char* in, char* out);
WritingFileStatus FILEMANAGER_WriteFile(Serial* serial, char* filename, char* input);
void FILEMANAGER_Deletefile(char filename[]);
uint32_t FILEMANAGER_GetRxID();
uint32_t FILEMANAGER_GetTxID();
#endif
