/* Copyright (c) 2021 Analog Devices, Inc.  All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright notice, 
  this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright notice, 
  this list of conditions and the following disclaimer in the documentation 
  and/or other materials provided with the distribution.  
  - Modified versions of the software must be conspicuously marked as such.
  - This software is licensed solely and exclusively for use with processors/products 
  manufactured by or for Analog Devices, Inc.
  - This software may not be combined or merged with other code in any manner 
  that would cause the software to become subject to terms and conditions which 
  differ from those listed here.
  - Neither the name of Analog Devices, Inc. nor the names of its contributors 
  may be used to endorse or promote products derived from this software without 
  specific prior written permission.
  - The use of this software may or may not infringe the patent rights of one or 
  more patent holders.  This license does not release you from the requirement 
  that you obtain separate licenses from these patent holders to use this software.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES, INC. AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT, 
TITLE, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN 
NO EVENT SHALL ANALOG DEVICES, INC. OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, DAMAGES ARISING OUT OF CLAIMS OF INTELLECTUAL 
PROPERTY RIGHTS INFRINGEMENT; PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS 
OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

2021-01-10-7CBSD SLA
*/


/*=============  I N C L U D E S   =============*/
#include "ADMX2001.h"
#include "ADMX2001_commands.h"
#include "platform_drivers.h"
#include "app_config.h"
#include "spi_extra.h"
#include "message.h"
#include "gpio_config.h"
#include "measure.h"
#include "calibrate.h"
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

/*=============  D A T A  =============*/

/** This is kept global and non-static as the test code requires this*/
Admx200xDev admx200xDev;

static int32_t SPIInit(spi_desc **pSpiDesc);

static int32_t ResetModule();

static int32_t SetGain(uint32_t voltageGain, uint32_t currentGain);

/*=============  C O D E  =============*/

 /**
 * @brief Function to initialize SPI
 * @param pSpiDesc - Pointer to the spi descriptor
 * @return Returns 0 for success or error code.
 */
int32_t SPIInit(spi_desc **pSpiDesc)
{
    int32_t status;

    // Init SPI extra parameters structure
    mbed_spi_init_param spiInitExtraParams = {.spi_clk_pin = SPI_SCK,
                                              .spi_miso_pin = SPI_MISO,
                                              .spi_mosi_pin = SPI_MOSI};

    spi_init_param spiInit = {
        25000000, // Max SPI Speed
        SPI_SS,  // Chip Select pin
        SPI_MODE_0,
        &spiInitExtraParams, // SPI extra configurations
    };

    status = spi_init(pSpiDesc, &spiInit);

    return status;
}

/**
 * @brief Resets measurement 
 * @return Returns 0 for success or error code.
 */
int32_t ResetModule()
{
    int32_t status = 0;
    uint8_t cmdId;
    uint16_t addr;
    uint32_t data;
    uint32_t statReg;

    Admx200xDev *pAdmx200x = &admx200xDev;

    /* Send reset command */
    cmdId = CMD_RESET;
    addr = 0;
    data = 0;
    status = Admx200xSendCmd(pAdmx200x, &cmdId, &addr, &data, &statReg);

    return status;
}

/**
 * @brief Set the voltage and current gain 
 * @param voltageGain - Voltage gain value
 * @param currentGain - Current gain value
 * @return Returns 0 for success or error code.
 */
int32_t SetGain(uint32_t voltageGain, uint32_t currentGain)
{
    int32_t status = ADMX_STATUS_SUCCESS;
    uint16_t addr = 0;
    uint8_t cmdId;
    uint32_t statReg;
    Admx200xDev *pAdmx200x = &admx200xDev;
    uint32_t data;
    
    data = voltageGain;
    cmdId = CMD_VOLTAGE_GAIN;
    status = Admx200xSendCmd(pAdmx200x, &cmdId, &addr, &data, &statReg);
    if (status == ADMX_STATUS_SUCCESS)
    {
        status = (int32_t)(statReg & ADMX200X_STATUS_CODE_BITM);
    }
    
    if (status == ADMX_STATUS_SUCCESS)
    {
        cmdId = CMD_CURRENT_GAIN;
        status = Admx200xSendCmd(pAdmx200x, &cmdId, &addr, &data, &statReg);
        if (status == ADMX_STATUS_SUCCESS)
        {
            status = (int32_t)(statReg & ADMX200X_STATUS_CODE_BITM);
        }
    }

    return status;
}

int main()
{
    int32_t status = 0;
    static spi_desc spiDesc;
    float stdLoad = 51;
    float Xt = 0;
    CAL_TYPE calType;
    spi_desc *pSpi = &spiDesc;
    Admx200xDev *pADmx2001 = &admx200xDev;
    status |= SPIInit(&pSpi);
    status |= Admx200xInit(pADmx2001, pSpi);

    mdelay(100);

    /* Initialize the ADMX200x application */
    if (status != 0)
    {
        ERROR_MSG("Setting up ADMX200X failed");
    }
    else
    {
        INFO_MSG("Resetting module");
        status = ResetModule();
        if (status != ADMX_STATUS_SUCCESS)
        {
            ERROR_MSG("Failed to reset module");
        }
        else
        {
            INFO_MSG("------ Initialising measurement ------");
            INFO_MSG("Setting test load to 1k ohms");
            // Selects test load 3 - 1000 ohm resistor
            EnableMuxLine(3);
            // Performing measurement
            status = MeasureAndDisplay();
            INFO_MSG("------ Measurement completed ------");
            
            INFO_MSG("------ Starting Calibration ------");
            INFO_MSG("Setting test load to short circuit");
            // Short the terminals across the lcr meter
            EnableMuxLine(1);
            // Setting the gain for calibration
            status = SetGain(0, 0);
            INFO_MSG("Setting voltage gain to zero");
            INFO_MSG("Setting current gain to zero");
            // Running short calibration
            calType = CAL_TYPE_SHORT;
            status = Calibrate(calType, stdLoad, Xt);
            if (status != ADMX_STATUS_SUCCESS)
            {
                ERROR_MSG("Calibration Failed");
            }
            INFO_MSG("------ Calibration completed ------");
        }
    }

    status = spi_remove(pSpi);

    return status;
}
