#include "mbed.h" // Include the library of Mbed
#include "LoRaWANInterface.h"
#include "SX1276_LoRaRadio.h"


/*
Send LoRa message to a the group project Thingspeak channel
https://thingspeak.com/channels/1234574/feed.json
 */

// The port we're sending and receiving on
#define MBED_CONF_LORA_APP_PORT     1

InterruptIn send_data_button(p6);

// Device credentials, register device as OTAA in The Things Network and copy credentials here
static uint8_t DEV_EUI[] = { 0x18, 0x11, 0x20, 0x20, 0x18, 0x11, 0x20, 0x20 };
static uint8_t APP_EUI[] = { 0x70, 0xB3, 0xD5, 0x7E, 0xD0, 0x03, 0x84, 0x15 };
static uint8_t APP_KEY[] = { 0x76, 0x8B, 0xD9, 0x68, 0xA9, 0x4C, 0xC0, 0xF6, 0x52, 0x20, 0x48, 0x7C, 0x00, 0xE6, 0x0D, 0x3C };


// Peripherals (LoRa radio, temperature sensor and button)
SX1276_LoRaRadio radio(D11, D12, D13, D10, A0, D2, D3, D4, D5, D8, D9, NC, NC, NC, NC, A4, NC, NC);

// EventQueue is required to dispatch events around
static EventQueue ev_queue;
// Constructing Mbed LoRaWANInterface and passing it down the radio object.
static LoRaWANInterface lorawan(radio);
// Application specific callbacks
static lorawan_app_callbacks_t callbacks;
// LoRaWAN stack event handler
static void lora_event_handler(lorawan_event_t event);

// Send a message over LoRaWAN
static void send_message(char *device, long int start, long int end, int topic) {
    uint8_t tx_buffer[50] = { 0 };
    // Sending strings over LoRaWAN is not recommended
    sprintf((char*) tx_buffer, "{\"device\":\"%s\",\"start\":%ld,\"end\":%ld,\"topic\":%d}", device, start, end, topic);
    int packet_len = strlen((char*) tx_buffer);
    printf("Sending %d bytes: \"%s\"\n", packet_len, tx_buffer);
    int16_t retcode = lorawan.send(MBED_CONF_LORA_APP_PORT, tx_buffer, packet_len, MSG_UNCONFIRMED_FLAG);
    // for some reason send() returns -1... I cannot find out why, the stack returns the right number. I feel that this is some weird Emscripten quirk
    if (retcode < 0) {
        retcode == LORAWAN_STATUS_WOULD_BLOCK ? printf("send - duty cycle violation\n")
                : printf("send() - Error code %d\n", retcode);
        return;
    }
    printf("%d bytes scheduled for transmission\n", retcode);
}

int initialize_lora() {
    if (DEV_EUI[0] == 0x0 && DEV_EUI[1] == 0x0 && DEV_EUI[2] == 0x0 && DEV_EUI[3] == 0x0 && DEV_EUI[4] == 0x0 && DEV_EUI[5] == 0x0 && DEV_EUI[6] == 0x0 && DEV_EUI[7] == 0x0) {
        printf("Set your LoRaWAN credentials first!\n");
        return -1;
    }
    if (lorawan.initialize(&ev_queue) != LORAWAN_STATUS_OK) {
        printf("LoRa initialization failed!\n");
        return -1;
    }
    // prepare application callbacks
    callbacks.events = mbed::callback(lora_event_handler);
    lorawan.add_app_callbacks(&callbacks);
    // Disable adaptive data rating
    if (lorawan.disable_adaptive_datarate() != LORAWAN_STATUS_OK) {
        printf("disable_adaptive_datarate failed!\n");
        return -1;
    }
    lorawan.set_datarate(5); // SF7BW125
    lorawan_connect_t connect_params;
    connect_params.connect_type = LORAWAN_CONNECTION_OTAA;
    connect_params.connection_u.otaa.dev_eui = DEV_EUI;
    connect_params.connection_u.otaa.app_eui = APP_EUI;
    connect_params.connection_u.otaa.app_key = APP_KEY;
    connect_params.connection_u.otaa.nb_trials = 3;

    lorawan_status_t retcode = lorawan.connect(connect_params);

    if (retcode == LORAWAN_STATUS_OK ||
        retcode == LORAWAN_STATUS_CONNECT_IN_PROGRESS) {
    } else {
        printf("Connection error, code = %d\n", retcode);
        return -1;
    }

    printf("LoRa Connection - In Progress ...\r\n");
    // make your event queue dispatching events forever
    ev_queue.dispatch_forever();
    return 0;
}

void send_data_button_isr() {

     char device[] = "MyDevice";
     int topic = 3; // topic identifier
     long int start = 1606037871;  // start reading
     long int end = 1606038051; // end reading

     send_message(device, start, end, topic);
}

int main() {
    send_data_button.rise(ev_queue.event(&send_data_button_isr)); // Registering an ISR for button click
    initialize_lora();
}

// Event handler
static void lora_event_handler(lorawan_event_t event) {
    switch (event) {
        case CONNECTED:
            printf("LoRA Connection - Successful\n");
            break;
        case DISCONNECTED:
            ev_queue.break_dispatch();
            printf("LoRa Disconnected Successfully\n");
            break;
        case TX_DONE:
            printf("Message Sent to Network Server\n");
            break;
        case TX_TIMEOUT:
        case TX_ERROR:
        case TX_CRYPTO_ERROR:
        case TX_SCHEDULING_ERROR:
            printf("Transmission Error - EventCode = %d\n", event);
            break;
        case JOIN_FAILURE:
            printf("OTAA Failed - Check Keys\n");
            break;
        default:
            MBED_ASSERT("Unknown Event");
    }
}


