#include "map.h"

#include "globals.h"
#include "graphics.h"

/**
 * The Map structure. This holds a HashTable for all the MapItems, along with
 * values for the width and height of the Map.
 */
struct Map {
    HashTable* items;
    int w, h;
};

/**
 * Storage area for the maps.
 * This is a global variable, but can only be access from this file because it
 * is static.
 */
static Map map[NUM_MAPS];
static int active_map;

/**
 * The first step in HashTable access for the map is turning the two-dimensional
 * key information (x, y) into a one-dimensional unsigned integer.
 * This function should uniquely map (x,y) onto the space of unsigned integers.
 */
static unsigned XY_KEY(int X, int Y, int size) {
    return (X+size*Y); //linearize a 2d index
}

/**
 * This is the hash function actually passed into createHashTable. It takes an
 * unsigned key (the output of XY_KEY) and turns it into a hash value (some
 * small non-negative integer).
 */
unsigned map_hash1(unsigned key)
{
    return key%NUM_BUCKETS1;
}

unsigned map_hash2(unsigned key)
{
    return key%NUM_BUCKETS2;
}

void maps_init()
{
    map[0].items = createHashTable(map_hash1, NUM_BUCKETS1);
    map[0].h = 50;
    map[0].w = 50;
    map[1].items = createHashTable(map_hash2, NUM_BUCKETS2);
    map[1].h = 20;
    map[1].w = 20;
}

Map* get_active_map()
{
    // There's only one map
    return &map[active_map];
}

Map* set_active_map(int m)
{
    active_map = m;
    return &map[active_map];
}


void print_map()
{
    // As you add more types, you'll need to add more items to this array.
    char lookup[] = {'W', 'P', 'N'};
    for(int y = 0; y < map_height(); y++)
    {
        for (int x = 0; x < map_width(); x++)
        {
            MapItem* item = get_here(x,y);
            if (item) pc.printf("%c", lookup[item->type]);
            else pc.printf(" ");
        }
        pc.printf("\r\n");
    }
}

int map_width()
{
    return map[active_map].w;
}

int map_height()
{
    return map[active_map].h;
}

int map_area()
{
    return map[active_map].h * map[active_map].w;
}

MapItem* get_north(int x, int y)
{
    return (MapItem*)getItem(map[active_map].items, XY_KEY(x,y-1,map_height()));
}

MapItem* get_south(int x, int y)
{
    return (MapItem*)getItem(map[active_map].items, XY_KEY(x,y+1,map_height()));
}

MapItem* get_east(int x, int y)
{
    return (MapItem*)getItem(map[active_map].items, XY_KEY(x+1,y,map_height()));
}

MapItem* get_west(int x, int y)
{
    return (MapItem*)getItem(map[active_map].items, XY_KEY(x-1,y,map_height()));
}

MapItem* get_here(int x, int y)
{
    return (MapItem*)getItem(map[active_map].items, XY_KEY(x,y, map_height()));
}


void map_erase(int x, int y)
{
    deleteItem(map[active_map].items, XY_KEY(x,y,map_height()));
}

void add_wall(int x, int y, int dir, int len)
{
    for(int i = 0; i < len; i++)
    {
        MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
        w1->type = WALL;
        w1->draw = draw_wall;
        w1->walkable = false;
        w1->data = NULL;
        unsigned key = (dir == HORIZONTAL) ? XY_KEY(x+i, y, map_height()) : XY_KEY(x, y+i, map_height());
        void* val = insertItem(get_active_map()->items, key, w1);
        if (val) free(val); // If something is already there, free it
    }
}

void add_plant(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = PLANT;
    w1->draw = draw_plant;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y, map_height()), w1);
    if (val) free(val); // If something is already there, free it
}

void add_plant2(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = PLANT2;
    w1->draw = draw_plant;
    w1->walkable = true;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y, map_height()), w1);
    if (val) free(val); // If something is already there, free it
}

void add_startNPC(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = STARTNPC;
    w1->draw = draw_startNPC;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y,map_height()), w1);
    if (val) free(val); // If something is already there, free it
}

void add_swordInStone(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = SWORDINSTONE;
    w1->draw = draw_swordInStone;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y, map_height()), w1);
    if (val) free(val); // If something is already there, free it
}

void add_elvarg(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = ELVARG;
    w1->draw = draw_elvarg;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y, map_height()), w1);
    if (val) free(val); // If something is already there, free it
}

void add_cave(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = CAVE;
    w1->draw = draw_cave;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y, map_height()), w1);
    if (val) free(val); // If something is already there, free it
}

void add_gate(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = GATE;
    w1->draw = draw_gate;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y, map_height()), w1);
    if (val) free(val); // If something is already there, free it
}

void add_treasure(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = TREASURE;
    w1->draw = draw_treasure;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y, map_height()), w1);
    if (val) free(val); // If something is already there, free it
}

void add_boulder(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = BOULDER;
    w1->draw = draw_boulder;
    w1->walkable = 2;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y, map_height()), w1);
    if (val) free(val); // If something is already there, free it
}

void add_phat(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = PHAT;
    w1->draw = draw_phat;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y, map_height()), w1);
    if (val) free(val); // If something is already there, free it
}

void add_rolling(int x, int y)
{
    MapItem* w1 = (MapItem*) malloc(sizeof(MapItem));
    w1->type = ROLLING;
    w1->draw = draw_rolling;
    w1->walkable = false;
    w1->data = NULL;
    void* val = insertItem(get_active_map()->items, XY_KEY(x, y, map_height()), w1);
    if (val) free(val); // If something is already there, free it
}