/*
MIT License

Copyright (c) 2018 Alexander Dölz

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

/*
   micro:drop - A simple game for the BBC micro:bit

   Fruits falling from the sky. Use buttons A & B to control the player
   and catch them all.

   Drop speed increases every 10 catches. How much can you get?
   
   After game over press AB to restart.
*/
#include "MicroBit.h"

struct Sprite {
    int     x;
    int     y;
};

const MicroBitImage imageCross("255,0,0,0,255\n0,255,0,255,0\n0,0,255,0,0\n0,255,0,255,0\n255,0,0,0,255\n");
const MicroBitImage imageFlash("255,255,255,255,255\n255,255,255,255,255\n255,255,255,255,255\n255,255,255,255,255\n255,255,255,255,255\n");

MicroBit uBit;
Sprite player;
Sprite fruit;

int game_over;
int paused;
int life;
int score;
int level;
int game_speed;
int high_score;
int new_high_score;

/* Move player sprite to the left */
void onButtonA(MicroBitEvent e)
{
    if (player.x > 0 && !paused && !game_over) {
        uBit.display.image.setPixelValue(player.x, player.y, 0);
        player.x--;
        uBit.display.image.setPixelValue(player.x, player.y, 255);
    }
}

/* Move player sprite to the right */
void onButtonB(MicroBitEvent e)
{
    if (player.x < 4 && !paused && !game_over) {
        uBit.display.image.setPixelValue(player.x, player.y, 0);
        player.x++;
        uBit.display.image.setPixelValue(player.x, player.y, 255);
    }
}

/* Move fruit down and check for missed fruits */
void dropFruit()
{
    while(!game_over) {
        if(!paused) {
            uBit.display.image.setPixelValue(fruit.x, fruit.y, 0);
            fruit.y++;
            // When the fruit is moved below the screen, it's a miss
            if (fruit.y > 4) {
                // Create a new fruit
                fruit.x=uBit.random(5);
                fruit.y=0;

                // Show a cross to indicate the miss
                uBit.display.print(imageCross);
                uBit.sleep(25);
                uBit.display.clear();

                // Reduce life
                life--;

                // No more lifes means game over
                if (life == 0) game_over = true;
            }
            uBit.display.image.setPixelValue(fruit.x, fruit.y, 255);
        }
        // Wait some time to give the human a chance
        uBit.sleep(game_speed);
    }
}

/* Update player sprite and check for catched fruits*/
void updatePlayer()
{
    while(!game_over) {
        uBit.display.image.setPixelValue(player.x, player.y, 255);
        // Check if the player catched the fruit
        if (player.x==fruit.x && player.y==fruit.y) {
            // Create a new fruit above the screen; dropFruit will do the rest
            fruit.x=uBit.random(5);
            fruit.y=-1;

            // Flash all LEDs to indicate the catch
            uBit.display.print(imageFlash);
            uBit.sleep(25);
            uBit.display.clear();

            // Score a point
            score++;

            // Level and speed will be incressed every 10 points
            if (score % 10 == 0) {
                // Pause the game during the "Next Level" message
                paused = true;
                level++;
                if (game_speed > 50) game_speed -= 50;
                uBit.display.clear();
                uBit.display.scroll("LEVEL:");
                uBit.display.scroll(level);
                // Bring Play back to the initial position
                uBit.display.image.setPixelValue(player.x, player.y, 0);
                player.x = 2;
                uBit.display.image.setPixelValue(player.x, player.y, 255);
                // Resume the game
                paused = false;
            }
        }
        // Give the CPU a rest
        uBit.sleep(25);
    }
}
/* Game over message */
void gameOver()
{
    // Concatenate everthiny to ensure it will be in the right order at async
    // output
    const int cScore = score;
    ManagedString msGameOver = ManagedString("GAME OVER! SCORE: ") +  ManagedString(cScore);
    if (new_high_score) msGameOver = msGameOver + ManagedString(" NEW HIGH SCORE!");

    uBit.display.scrollAsync(msGameOver);
}

/* The Game */
void dropGame()
{
    // Show current high score
    if (high_score > 0) {
        uBit.display.scroll("HIGH SCORE:");
        uBit.display.scroll(high_score);
    }
    uBit.display.scroll("LEVEL: 1");
    
    // Initialize player sprite
    player.x=2;
    player.y=4;
    uBit.display.image.setPixelValue(player.x, player.y, 255);

    // Initialize fruit sprite above the screen; dropFruit will do the rest
    fruit.x=uBit.random(5);
    fruit.y=-1;

    // Reset game state
    paused = false;
    game_speed = 400;
    level = 1;
    score = 0;
    life = 3;
    new_high_score = false;
    game_over = false;

    //Create background tasks for player and fruit update
    create_fiber(updatePlayer);
    create_fiber(dropFruit);

    // Game logik is in fibers updatePlayer and dropFruit
    // So, while playing do nothing
    while(!game_over)
        uBit.sleep(200);

    // Check for new high score
    if (score > high_score) {
        high_score = score;
        // Maker it persistent
        uBit.storage.put("high_score", (uint8_t *)&high_score, sizeof(int));
        new_high_score = true;
    }

    // Show "Game Over" screen and score until A&B ist pressed together
    uBit.display.clear();

    while(!uBit.buttonAB.isPressed()) {
        gameOver();
        uBit.sleep(200);
    }
    uBit.display.stopAnimation();
}

int main(void)
{
    // Initialize the micro:bit runtime
    uBit.init();

    // Game titel
    uBit.display.scroll("MICRO:DROP");

    // Get current high score from storage
    KeyValuePair* stroredHighScore = uBit.storage.get("high_score");
    if(stroredHighScore  == NULL) {
        high_score = 0; // Seems to be the first boot; there is no high score
    } else {
        // Copy high score from key value pair into int var
        memcpy(&high_score, stroredHighScore->value, sizeof(int));
        delete stroredHighScore;
    }

    //Disable buttons & background tasks at startup
    game_over = true;

    //Register functions to buttons A & B
    uBit.messageBus.listen(MICROBIT_ID_BUTTON_A, MICROBIT_BUTTON_EVT_DOWN, onButtonA);
    uBit.messageBus.listen(MICROBIT_ID_BUTTON_B, MICROBIT_BUTTON_EVT_DOWN, onButtonB);

    // Play forever ;-)
    while(1) {
        dropGame();
    }
}