/*
    Sample program to use IS color switch from NKK.
    Part of this program is derived from a sample program for PIC
    micro cotnroller written by NKK.
*/
#include "mbed.h"
#include "SDFileSystem.h"

SDFileSystem sd(p5, p6, p7, p8, "sd");
SPI spi(p11, p12, p13); // mosi(SDO), miso(SDI), sclk(SCLK)
DigitalOut is_cs(p21);
DigitalOut is_reset(p22);
DigitalOut is_dc(p23);
DigitalOut is_vcc(p24);
DigitalOut led1(LED1);

void    OledAdrInit( int DevType );
void    OledDspMode( int DspMode );
void    OledInit( void );
void    OledSendCommand( const unsigned char *com );
void    OledSendData( const unsigned char *dat, int len );

const int DAT = 1;
const int CMD = 0;
int DevType = 1; // 0:ISC01, 1:ISC15
int DspMode = 1; // 0:256, 1:64k colors

unsigned char dat[64*48*2];
const unsigned char cls[64*48*2] = {0};

int main() {
    // Setup the spi for 8 bit data, high steady state clock,
    // second edge capture, with a 4MHz clock rate
    spi.format(8, 3);
    spi.frequency(4000000);

    OledInit();

#if 1
    for (;;) {
        DIR *d;
        struct dirent *p;
        d = opendir("/sd");
        if (d != NULL) {
            while ((p = readdir(d)) != NULL) {
                char fname[20];
                sprintf(fname, "/sd/%s", p->d_name);
                FILE *fp = fopen(fname, "rb");
                if (fp == NULL) {
                    // error("Could not open file for read\n");
                } else {
                    fread(dat, 1, 8, fp); // skip header
                    fread(dat, 1, sizeof(dat), fp);
                    fclose(fp);
                    OledAdrInit(DevType);
                    OledDspMode(DspMode);
                    OledSendData(dat, 64*48*2);
                }
            }
            closedir(d);
        }
    }
#else
    for (int i=0; i<64*48; i ++) {
        dat[i*2] = (i & 0xff00) >> 8;
        dat[i*2 + 1] = i & 0xff;
    }
    OledAdrInit(DevType);
    OledDspMode(DspMode);
    OledSendData(dat, 64*48*2);

    for (;;) {
        OledAdrInit(DevType);
        OledDspMode(DspMode);
        OledSendData(dat, 64*48*2);
        led1 = 0;
        wait (0.5);
        OledAdrInit(DevType);
        OledDspMode(DspMode);
        OledSendData(cls, 64*48*2);
        led1 = 1;
        wait (0.5);
    }
#endif
}

//    OLED's initial setup commands (IS15)
const    unsigned char    SetupColumnAddress[]        =    {    0x03,    0x15,    0x10,    0x4f    };
const    unsigned char    SetupRowaddress[]            =    {    0x03,    0x75,    0x00,    0x2f    };
const    unsigned char    SetContrastColorA[]         =    {    0x02,    0x81,    0x19    };
const    unsigned char    SetContrastColorB[]         =    {    0x02,    0x82,    0x14    };
const    unsigned char    SetContrastColorC[]         =    {    0x02,    0x83,    0x24    };
const    unsigned char    MasterCurrentControl[]        =    {    0x02,    0x87,    0x0f    };
const    unsigned char    RemapColorDepth[]            =    {    0x02,    0xa0,    0x70    };
const    unsigned char    SetDisplayStartLine[]        =    {    0x02,    0xa1,    0x00    };
const    unsigned char    SetDisplayOffset[]            =    {    0x02,    0xa2,    0x10    };
const    unsigned char    SetDisplayMode[]            =    {    0x01,    0xa4    };
const    unsigned char    SetMultiplexRatio[]            =    {    0x02,    0xa8,    0x2f    };
const    unsigned char    DimModeSetting[]            =    {    0x05,    0xab,    0x12,    0x0c,    0x14,    0x12    };
const    unsigned char    SetDisplayDim[]                =    {    0x01,    0xac    };
const    unsigned char    SetDisplayOff[]                =    {    0x01,    0xae    };
const    unsigned char    SetDisplayNormal[]            =    {    0x01,    0xaf    };
const    unsigned char    SetMasterConfiguration[]    =    {    0x02,    0xad,    0x8e    };
const    unsigned char    PhasePeriodAdjustment[]        =    {    0x02,    0xb1,    0x44    };
const    unsigned char    DisplayClockDivider[]        =    {    0x02,    0xb3,    0xa0    };
const    unsigned char    EnableLinearGrayScale[]        =    {    0x01,    0xb9    };
const    unsigned char    SetPrechargeLevel[]            =    {    0x02,    0xbb,    0x12    };
const    unsigned char    SetVcomh[]                    =    {    0x03,    0xbe,    0x28    };

const    unsigned char    ColorDepth64k[]                =    {    0x02,    0xa0,    0x70    };
const    unsigned char    ColorDepth256[]                =    {    0x02,    0xa0,    0x30    };

//    OLED's initial setup commands (for display only type (IS01) )
const    unsigned char    SetupColumnAddress_d[]        =    {    0x03,    0x15,    0x16,    0x49    }; //display
const    unsigned char    SetupRowaddress_d[]            =    {    0x03,    0x75,    0x00,    0x23    }; //display
const    unsigned char    SetContrastColorA_d[]         =    {    0x02,    0x81,    0x0f    }; //display
const    unsigned char    SetContrastColorB_d[]         =    {    0x02,    0x82,    0x0e    }; //display
const    unsigned char    SetContrastColorC_d[]         =    {    0x02,    0x83,    0x1b    }; //display
const    unsigned char    SetDisplayOffset_d[]            =    {    0x02,    0xa2,    0x1c    }; //display
const    unsigned char    SetMultiplexRatio_d[]            =    {    0x02,    0xa8,    0x23    }; //display
const    unsigned char    DimModeSetting_d[]            =    {    0x05,    0xab,    0x0b,    0x08,    0x0f,    0x12    }; //display
const    unsigned char    DisplayClockDivider_d[]        =    {    0x02,    0xb3,    0x30    }; //display
const    unsigned char    SetVcomh_d[]                    =    {    0x03,    0xbe,    0x21    }; //display
//
//        Change OLED's address setting
//
void    OledAdrInit( int DevType )    {
    if (DevType) {
        OledSendCommand( SetupColumnAddress );
        OledSendCommand( SetupRowaddress );
    } else {
        OledSendCommand( SetupColumnAddress_d );
        OledSendCommand( SetupRowaddress_d );
    }
}

//
//        Change OLED's color depth setting (64K or 256 colors)
//
void    OledDspMode( int DspMode )    {
    if ( DspMode )    {
        OledSendCommand( ColorDepth64k );
    } else    {
        OledSendCommand( ColorDepth256 );
    }
}

//
//        initial settings of OLED
//
void    OledInit( void )    {
    volatile unsigned char    delay = 0x20;

    //    reset OLED
    is_vcc = 0;
    is_reset = 0;
    wait_us(3);
    is_reset = 1;
    is_vcc = 1;

    //    initial settings
    OledAdrInit(DevType);

    if (DevType) {
        OledSendCommand( SetContrastColorA );
        OledSendCommand( SetContrastColorB );
        OledSendCommand( SetContrastColorC );
        OledSendCommand( MasterCurrentControl );
        OledSendCommand( SetDisplayStartLine );
        OledSendCommand( SetDisplayOffset );
        OledSendCommand( SetDisplayMode );
        OledSendCommand( SetMultiplexRatio );
        OledSendCommand( DimModeSetting );
        OledSendCommand( SetMasterConfiguration );
        OledSendCommand( PhasePeriodAdjustment );
        OledSendCommand( DisplayClockDivider );
        OledSendCommand( EnableLinearGrayScale );
        OledSendCommand( SetPrechargeLevel );
        OledSendCommand( SetVcomh );
    } else {
        OledSendCommand( SetContrastColorA_d );
        OledSendCommand( SetContrastColorB_d );
        OledSendCommand( SetContrastColorC_d );
        OledSendCommand( MasterCurrentControl );
        OledSendCommand( SetDisplayStartLine );
        OledSendCommand( SetDisplayOffset_d );
        OledSendCommand( SetDisplayMode );
        OledSendCommand( SetMultiplexRatio_d );
        OledSendCommand( DimModeSetting_d );
        OledSendCommand( SetMasterConfiguration );
        OledSendCommand( PhasePeriodAdjustment );
        OledSendCommand( DisplayClockDivider_d );
        OledSendCommand( EnableLinearGrayScale );
        OledSendCommand( SetPrechargeLevel );
        OledSendCommand( SetVcomh_d );
    }

    //    display mode settings
    OledDspMode(DspMode);

    OledSendCommand( SetDisplayNormal );
}

//
//      Send a command / data to OLED via SPI
//
void    OledSend( const unsigned char *p, int len )    {
    for (; len>0; len --, p ++) {
        is_cs = 0;
        spi.write(*p);
        is_cs = 1;
    }
}

//
//        Send a command to OLED
//
void    OledSendCommand( const unsigned char *com )    {
    is_dc = CMD;
    OledSend(com+1, *com);
}

//
//        Send data to OLED
//
void    OledSendData( const unsigned char *dat, int len )    {
    is_dc = DAT;
    OledSend(dat, len);
}
