/* mbed NKK IS-C15/C01 oLED switch library
 * Copyright (c) 2011, Noriaki Mitsunaga
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "isc15.h"

////////////////////////////////////////////////////////////////////////////////////////
// Some constants
////////////////////////////////////////////////////////////////////////////////////////
const int DAT = 1;
const int CMD = 0;

////////////////////////////////////////////////////////////////////////////////////////
//    OLED's initial setup commands (IS15)
////////////////////////////////////////////////////////////////////////////////////////
const    unsigned char    SetupColumnAddress[]        =    {    0x03,    0x15,    0x10,    0x4f    };
const    unsigned char    SetupRowaddress[]            =    {    0x03,    0x75,    0x00,    0x2f    };
const    unsigned char    SetContrastColorA[]         =    {    0x02,    0x81,    0x19    };
const    unsigned char    SetContrastColorB[]         =    {    0x02,    0x82,    0x14    };
const    unsigned char    SetContrastColorC[]         =    {    0x02,    0x83,    0x24    };
const    unsigned char    MasterCurrentControl[]        =    {    0x02,    0x87,    0x0f    };
const    unsigned char    RemapColorDepth[]            =    {    0x02,    0xa0,    0x70    };
const    unsigned char    SetDisplayStartLine[]        =    {    0x02,    0xa1,    0x00    };
const    unsigned char    SetDisplayOffset[]            =    {    0x02,    0xa2,    0x10    };
const    unsigned char    SetDisplayMode[]            =    {    0x01,    0xa4    };
const    unsigned char    SetMultiplexRatio[]            =    {    0x02,    0xa8,    0x2f    };
const    unsigned char    DimModeSetting[]            =    {    0x05,    0xab,    0x12,    0x0c,    0x14,    0x12    };
const    unsigned char    SetDisplayDim[]                =    {    0x01,    0xac    };
const    unsigned char    SetDisplayOff[]                =    {    0x01,    0xae    };
const    unsigned char    SetDisplayNormal[]            =    {    0x01,    0xaf    };
const    unsigned char    SetMasterConfiguration[]    =    {    0x02,    0xad,    0x8e    };
const    unsigned char    PhasePeriodAdjustment[]        =    {    0x02,    0xb1,    0x44    };
const    unsigned char    DisplayClockDivider[]        =    {    0x02,    0xb3,    0xa0    };
const    unsigned char    EnableLinearGrayScale[]        =    {    0x01,    0xb9    };
const    unsigned char    SetPrechargeLevel[]            =    {    0x02,    0xbb,    0x12    };
const    unsigned char    SetVcomh[]                    =    {    0x03,    0xbe,    0x28    };

const    unsigned char    ColorDepth64k[]                =    {    0x02,    0xa0,    0x70    };
const    unsigned char    ColorDepth256[]                =    {    0x02,    0xa0,    0x30    };

//    OLED's initial setup commands (for display only type (IS01) )
const    unsigned char    SetupColumnAddress_d[]        =    {    0x03,    0x15,    0x16,    0x49    }; //display
const    unsigned char    SetupRowaddress_d[]            =    {    0x03,    0x75,    0x00,    0x23    }; //display
const    unsigned char    SetContrastColorA_d[]         =    {    0x02,    0x81,    0x0f    }; //display
const    unsigned char    SetContrastColorB_d[]         =    {    0x02,    0x82,    0x0e    }; //display
const    unsigned char    SetContrastColorC_d[]         =    {    0x02,    0x83,    0x1b    }; //display
const    unsigned char    SetDisplayOffset_d[]            =    {    0x02,    0xa2,    0x1c    }; //display
const    unsigned char    SetMultiplexRatio_d[]            =    {    0x02,    0xa8,    0x23    }; //display
const    unsigned char    DimModeSetting_d[]            =    {    0x05,    0xab,    0x0b,    0x08,    0x0f,    0x12    }; //display
const    unsigned char    DisplayClockDivider_d[]        =    {    0x02,    0xb3,    0x30    }; //display
const    unsigned char    SetVcomh_d[]                    =    {    0x03,    0xbe,    0x21    }; //display

//
//        Change OLED's address setting
//
void    ISC15::AdrInit( int DevType )    {
    if (spi == NULL)
        return;

    if (DevType == ISC15_DEVICE_ISC15) {
        SendCommand( SetupColumnAddress );
        SendCommand( SetupRowaddress );
    } else {
        SendCommand( SetupColumnAddress_d );
        SendCommand( SetupRowaddress_d );
    }
}


//
//        Display data on OLED
//
void    ISC15::Disp( int Mode, const unsigned char *dat )    {
    if (spi == NULL)
        return;

    AdrInit(DevType);
    DspMode(Mode);
    if (Mode == ISC15_DSPMODE_64K)
        SendData(dat, ISC15_WIDTH*ISC15_HEIGHT*2);
    else
        SendData(dat, ISC15_WIDTH*ISC15_HEIGHT);
}

//
//        Change OLED's color depth setting (64K or 256 colors)
//
void    ISC15::DspMode( int DspMode )    {
    if (spi == NULL)
        return;

    if ( DspMode == ISC15_DSPMODE_64K )    {
        SendCommand( ColorDepth64k );
    } else    {
        SendCommand( ColorDepth256 );
    }
}

//
//        Fill out display
//
void    ISC15::Fill(unsigned char c)    {
    if (spi == NULL)
        return;

    AdrInit(DevType);
    DspMode(0);

    is_dc = DAT;
    for (int i = ISC15_WIDTH*ISC15_HEIGHT; i>0; i --) {
        is_cs = 0;
        spi->write(c);
        is_cs = 1;
    }
}

//
//        initial settings of OLED
//
void    ISC15::Init( int DeviceType, SPI *s )    {
    volatile unsigned char    delay = 0x20;

    DevType = DeviceType;
    spi = s;

    //    reset OLED
    is_vcc = 0;
    is_reset = 0;
    wait_us(3);
    is_reset = 1;
    is_vcc = 1;

    //    initial settings
    AdrInit(DevType);

    if (DevType == ISC15_DEVICE_ISC15) {
        SendCommand( SetContrastColorA );
        SendCommand( SetContrastColorB );
        SendCommand( SetContrastColorC );
        SendCommand( MasterCurrentControl );
        SendCommand( SetDisplayStartLine );
        SendCommand( SetDisplayOffset );
        SendCommand( SetDisplayMode );
        SendCommand( SetMultiplexRatio );
        SendCommand( DimModeSetting );
        SendCommand( SetMasterConfiguration );
        SendCommand( PhasePeriodAdjustment );
        SendCommand( DisplayClockDivider );
        SendCommand( EnableLinearGrayScale );
        SendCommand( SetPrechargeLevel );
        SendCommand( SetVcomh );
    } else {
        SendCommand( SetContrastColorA_d );
        SendCommand( SetContrastColorB_d );
        SendCommand( SetContrastColorC_d );
        SendCommand( MasterCurrentControl );
        SendCommand( SetDisplayStartLine );
        SendCommand( SetDisplayOffset_d );
        SendCommand( SetDisplayMode );
        SendCommand( SetMultiplexRatio_d );
        SendCommand( DimModeSetting_d );
        SendCommand( SetMasterConfiguration );
        SendCommand( PhasePeriodAdjustment );
        SendCommand( DisplayClockDivider_d );
        SendCommand( EnableLinearGrayScale );
        SendCommand( SetPrechargeLevel );
        SendCommand( SetVcomh_d );
    }

    //    display mode settings
    DspMode( ISC15_DSPMODE_64K);

    SendCommand( SetDisplayNormal );
}

//
//      Send a command / data to OLED via SPI
//
void    ISC15::Send( const unsigned char *p, int len )    {
    if (spi == NULL)
        return;

    for (; len>0; len --, p ++) {
        is_cs = 0;
        spi->write(*p);
        is_cs = 1;
    }
}

//
//        Send a command to OLED
//
void    ISC15::SendCommand( const unsigned char *com )    {
    if (spi == NULL)
        return;

    is_dc = CMD;
    Send(com+1, *com);
}

//
//        Send data to OLED
//
void    ISC15::SendData( const unsigned char *dat, int len )    {
    if (spi == NULL)
        return;

    is_dc = DAT;
    Send(dat, len);
}
