// original: http://developer.mbed.org/users/ytsuboi/code/BLE_HTM_HRM1017/shortlog
// modified for ADT7410

#include "mbed.h"
#include "BLEDevice.h"
#include "ble_hts.h"

#define NEED_CONSOLE_OUTPUT 1 /* Set this if you need debug messages on the console;
                               * it will have an impact on code-size and power consumption. */

#if NEED_CONSOLE_OUTPUT
Serial  pc(USBTX, USBRX);
#define DEBUG(...) { pc.printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

const static char  DEVICE_NAME[] = "HRM1017_HTM";
static volatile bool  triggerSensorPolling = false;

BLEDevice  ble;

#define UseADT7410 
#ifdef UseADT7410
#include "ADT7410.h"
//ADT7410 healthThemometer(p13, p15, 0x90, 100000);
ADT7410 healthThemometer(p22, p20, 0x90, 100000);
#else
#include "TMP102.h"
TMP102      healthThemometer(p22, p20, 0x90);  /* The TMP102 connected to our board */
#endif

/* LEDs for indication: */
DigitalOut  oneSecondLed(LED1);        /* LED1 is toggled every second. */
DigitalOut  advertisingStateLed(LED2); /* LED2 is on when we are advertising, otherwise off. */


/* Health Thermometer Service */ 
uint8_t             thermTempPayload[5] = { 0, 0, 0, 0, 0 };
GattCharacteristic  tempChar (GattCharacteristic::UUID_TEMPERATURE_MEASUREMENT_CHAR,
                                thermTempPayload, 5, 5,
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_INDICATE);
GattCharacteristic *htmChars[] = {&tempChar, };
GattService        htmService(GattService::UUID_HEALTH_THERMOMETER_SERVICE, htmChars, 
                                sizeof(htmChars) / sizeof(GattCharacteristic *));

/* Battery Level Service */
uint8_t            batt = 100;     /* Battery level */
uint8_t            read_batt = 0;  /* Variable to hold battery level reads */
GattCharacteristic battLevel ( GattCharacteristic::UUID_BATTERY_LEVEL_CHAR, 
                                 (uint8_t *)batt, 1, 1,
                                 GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);
GattCharacteristic *battChars[] = {&battLevel, };
GattService        battService(GattService::UUID_BATTERY_SERVICE, battChars,
                                sizeof(battChars) / sizeof(GattCharacteristic *));

uint16_t             uuid16_list[] = {GattService::UUID_HEALTH_THERMOMETER_SERVICE,
                                      GattService::UUID_BATTERY_SERVICE};

uint32_t quick_ieee11073_from_float(float temperature);
void updateServiceValues(void);

static Gap::ConnectionParams_t connectionParams;

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)    // Mod
{
    advertisingStateLed = 1;
    
    DEBUG("Disconnected handle %u, reason %u\n", handle, reason);
    DEBUG("Restarting the advertising process\n\r");
    ble.startAdvertising();
}

void onConnectionCallback(Gap::Handle_t handle, const Gap::ConnectionParams_t *params)   //Mod
{
    advertisingStateLed = 0;

    DEBUG("connected. Got handle %u\r\n", handle);

    connectionParams.slaveLatency = 1;
    if (ble.updateConnectionParams(handle, &connectionParams) != BLE_ERROR_NONE) {
        DEBUG("failed to update connection paramter\r\n");
    }
}

void periodicCallback(void)
{
    oneSecondLed = !oneSecondLed; /* Do blinky on LED1 while we're waiting for BLE events */

    /* Note that the periodicCallback() executes in interrupt context, so it is safer to do
     * heavy-weight sensor polling from the main thread. */
    triggerSensorPolling = true;
}

/**************************************************************************/
/*!
    @brief  Program entry point
*/
/**************************************************************************/
int main(void)
{
    
    /* Setup blinky led */
    oneSecondLed = 1;
    Ticker ticker;
    ticker.attach(periodicCallback, 1);
       
    DEBUG("Initialising the nRF51822\n");
    ble.init();
    DEBUG("Init done\n");
    ble.onDisconnection(disconnectionCallback);
    ble.onConnection(onConnectionCallback);

    ble.getPreferredConnectionParams(&connectionParams);

    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t*)uuid16_list, sizeof(uuid16_list));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_THERMOMETER);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.startAdvertising();
    advertisingStateLed = 1;
    DEBUG("Start Advertising\n");

    ble.addService(htmService);
    ble.addService(battService);
    DEBUG("Add Service\n");
    
    healthThemometer.reset();
    //healthThemometer.setConfig(ONE_SPS_MODE);

    while (true) {
        if (triggerSensorPolling) {
            triggerSensorPolling = false;
            updateServiceValues();
        } else {
            ble.waitForEvent();
        }
    }
}

/**************************************************************************/
/*!
    @brief  Ticker callback to switch advertisingStateLed state
*/
/**************************************************************************/
#include <stdlib.h>
void updateServiceValues(void)
{
      /* Decrement the battery level. */
      batt <=50 ? batt=100 : batt--;
      
      /* Update the temperature. Note that we need to convert to an ieee11073 format float. */
#ifdef UseADT7410
    float temperature = healthThemometer.getTemp();
#else
      float temperature = healthThemometer.read();
#endif
      DEBUG("temp:%f\n", temperature);
      uint32_t temp_ieee11073 = quick_ieee11073_from_float(temperature);
      memcpy(thermTempPayload+1, &temp_ieee11073, 4);
      ble.updateCharacteristicValue(tempChar.getValueAttribute().getHandle(), thermTempPayload, sizeof(thermTempPayload));  //Mod
      ble.updateCharacteristicValue(battLevel.getValueAttribute().getHandle(), (uint8_t *)&batt, sizeof(batt));             //Mod
}

/**
 * @brief A very quick conversion between a float temperature and 11073-20601 FLOAT-Type.
 * @param temperature The temperature as a float.
 * @return The temperature in 11073-20601 FLOAT-Type format.
 */
uint32_t quick_ieee11073_from_float(float temperature)
{
    uint8_t  exponent = 0xFF; //exponent is -1
    uint32_t mantissa = (uint32_t)(temperature*10);
    
    return ( ((uint32_t)exponent) << 24) | mantissa;
}

