/**----------------------------------------------------------------------------
 *
 *            \file frequency_detector.cpp
--                                                                           --
--              ECEN 5803 Mastering Embedded System Architecture             --
--                  Project 1 Module 4                                       --
--                Microcontroller Firmware                                   --
--                      frequency_detector.cpp                                           --
--                                                                           --
-------------------------------------------------------------------------------
--
--  Designed for:  University of Colorado at Boulder
--                
--                
--  Designed by:  Tim Scherr
--  Revised by:  Naved Maududi and Bryan Cisneros 
-- 
-- Version: 2.1
-- Date of current revision:  2017-09-25   
-- Target Microcontroller: Freescale MKL25ZVMT4 
-- Tools used:  ARM mbed compiler
--              ARM mbed SDK
--              Freescale FRDM-KL25Z Freedom Board
--               
--               
   Functional Description:  
   This file contains calcuate the flow of fluid that is being measured. The flow i
     measured in gallons per minute. The flow function takes in two inputs, which is the
     frequency and temperature. Frequency is calculated from the vortex analog signal on 
     ADC channel 2. The temperature is determined from an internal sensor. 
-- 
--      Copyright (c) 2015 Tim Scherr  All rights reserved.
*/


#include "shared.h" 
//#include <math.h>

#define BLUFF_BODY_INCHES (0.5)
#define BLUFF_BODY_METERS (0.0127)
#define PID_INCHES (2.9)
#define PID_INCHES_SQUARED (8.41)
#define PID_METERS (0.07366)
#define INCHES_IN_METER (39.37)



float calculateViscosity(uint16_t temperature)
{
    float power = 247.8/((float)temperature - 140.0);
    return (float)0.000024 * pow(10,(power));
}

float calculateDensity(uint16_t temperature)
{
    return 1000 * (1 - (((float)temperature+288.9414)/(508929.2*((float)temperature+68.12963)))*pow(((float)temperature-3.9863),2));
}

float calculateReynoldsNumber(float density, float viscosity, float velocity)
{
    return density * velocity * PID_METERS / viscosity;
}

float calculateStrouhalNumber(float reynoldsNumber)
{
    return 0.2684 - 1.0356/sqrt(reynoldsNumber);
}

float calculateVelocity(uint16_t frequency, float strouhalNumber)
{
    return (float)frequency * BLUFF_BODY_METERS / strouhalNumber;
}

float convertVelocityMetersToInches(float velocityMeters)
{
    return velocityMeters * INCHES_IN_METER;
}

void calculateFlow(void)
{
    float velocity_prev = 0;
    float velocity = 10;
    float velocity_dif = 0.1;
    float strouhalNumber = 0;
    float reynoldsNumber = 0;
    float density = 0;
    float viscosity = 0;

    while (velocity_dif > 0.01)
    {
        velocity_prev = velocity;
        viscosity = calculateViscosity(temperature);
        //printf("viscosity: %f\r\n", viscosity);
        density = calculateDensity(temperature);
        //printf("density: %f\r\n", density);
        reynoldsNumber = calculateReynoldsNumber(density, viscosity, velocity_prev);
        //printf("reynoldsNumber: %f\r\n", reynoldsNumber);
        strouhalNumber = calculateStrouhalNumber(reynoldsNumber);
        //printf("strouhalNumber: %f\r\n", strouhalNumber);
        velocity = calculateVelocity(freq_value, strouhalNumber);
        velocity_dif = fabs((velocity - velocity_prev)/(velocity + velocity_prev));
        //printf("velocity: %f\r\n", velocity);
    }

    flow =  (2.45 * PID_INCHES_SQUARED * velocity);
}
