//***************************************************************************//
/* InterruptMask Library */
/* Napoleon Leoni, February 18,2014*/
/* This class enables masking a GPIO interrupt from a specific MBED pin*/
// Note to get the actual pin number in the LPC of an mbed pin in P0
// simply do pN-P0_0, for example p12-P0_0=17, that is pin 12 in the mbed
// corresponds to Port[0].17 or bit 17 of port 0.
//***************************************************************************//
//* Example usage */
//* >#include "InterruptMask.h"
//* >InterruptIn upPressed(p15);
//* >InterruptMask maskUpButtonInt(p15); //Interrupt mask object allows easy access to 
//* >                                    //mask/unmask interrupts from a GPIO source
//* >void ISRup(void){
//*>    myled1=!myled1;
//*>}
//*>int main() {
//*>    upPressed.rise(&ISRup);
//*>    //downPressed.rise(&ISRdown);
//*>    bool disableInterrupt=false;
//*>    int counter=0,tdelay=10;
//*>    
//*>    while (1) {
//*>        //This section of code periodically (every 10 seconds) masks the specified GPIO interrupt
//*>        //showing how to both mask it and unmask it as well as how to clear pending interrupts
//*>        counter++;
//*>        if(counter>=tdelay){
//*>            disableInterrupt=!disableInterrupt;
//*>            if(disableInterrupt) maskUpButtonInt.maskIntR() ;   //mask interrupt
//*>            if(!disableInterrupt){
//*>                 maskUpButtonInt.ClrInt();       //clear any pending interrupt before unmasking
//*>                 maskUpButtonInt.unMaskIntR();   //unmask interrupt 
//*>            }
//*>            counter=0;
//*>        }
//*>    }
//*>}

//
//                  TESTS RUN                                               //
// I print to the LCD three important LPC1768 registers: 
// ISER0..Interrupt set Enable register(address 0xE000E100), 
// contents of ISER0 are accesible via: *(NVIC->ISER)
//
// IntEnR..GPIO interrupt enable on rising, shows which pins are attached to EINT3 
// on their rising edge (these are only for mbed pins mapped to LPC pins on Port 0)
// its contents are available using th macro LPC_GPIOINT->IO0IntEnR
// IntEnF..GPIO interrupt enable on falling, shows which pins are attached to EINT3 
// on their falling edge (these are only for mbed pins mapped to LPC pins on Port 0)
// its contents are available using th macro LPC_GPIOINT->IO0IntEnF
//
// The test program periodically masks/unmasks (every 10 seconds) a specified interrupt
// from a GPIO pin, 
// 
//***********************************************************************************
//   TEST DESCRIPTION                 Resulting Register Values                     ANALYSYS
// 1 ISR attached to rise             ISER0       0x0020 0010                       Works as expected when p15 interrupt on rise edge
//   edge of p15(toggle LED1)         IO0IntEnR   0x0082 0000 (when p15 unmasked)   is masked the proper bit appears masked in IO0IntEnR
//   and p12(toggle LED2)             IO0IntEnR   0x0002 0000 (when p15 masked)     and the up action of the joystick does not toggle the LED
//                                                                                  while the down motion continues to work. Once the interrupt
//                                                                                  from p15 is unmasked the up joysstick action works again.
//   TEST DESCRIPTION                                           STATUS
// 2 check mask functionality for all                           PASS
//   GPIO pins attached to joystick , both
//   for rise and fall attached interrupts
//
// 3 When interrupt is cleared (ClrInt() )                      PASS
//   before unmasking there should be no pending 
//   interrupts
//   
//***********************************************************************************

#include "mbed.h"

#ifndef INTERRUPTMASK
#define INTERRUPTMASK

class InterruptMask{
    
public:

private:
    int mPin;
    int mask;
    int unmask;
    volatile unsigned int* IOIntEnF; // pointer to register with fall enable interrupt for 
                                     // appropriate LPC port depending on the GPIO pin
    volatile unsigned int* IOIntEnR;// pointer to register with rise enable interrupt for 
                                     // appropriate LPC port depending on the GPIO pin
    volatile unsigned int* IOIntClr; //pointer to register to clear interrupt for 
                                     // appropriate LPC port depending on the GPIO pin                                
public:
    //default constructor required, should not be use will yield error
    InterruptMask();
    //Constructor to be used, initialize with any valid mbed pin (p5 thru p30 except pins 19-20)
    //you will need one InterruptMask for every pin you want ot have control over.
    InterruptMask(PinName GPIOpin);
    
    //This method will mask (disable) any rising edge interrupts of the specified pin
    void maskIntR(void);
    
    //This method will mask (disable) any faling edge interrupts of the specified pin
    void maskIntF(void);

    //This method will unmask (enable) any rising edge interrupts of the specified pin
    void unMaskIntR(void);

    //This method will unmask (enable) any falling edge interrupts of the specified pin
    void unMaskIntF(void);
      
    //This method will clear any pending interrupts of the specified pin
    void ClrInt(void);

};

#endif
