/* DebounceInt2 Library */
/* This class enables debouncing a GPIO interrupt from a specific MBED pin            */
/* It does so by using the InterruptMask library to temporarily mask the interrupt    */
/* from the specified pin for a user specified delay specified in us. After the delay */
/* time has passed the interrupt is unmasked (reenabled) and any pending interrupts   */
/* also cleared. The library gives you the flexibiity to debounce on either the       */
/* rising or falling edge interrupts.                                                 */
// Deprecated Usage:
// Declare a DebounceInt2 object with a valid mbed pin (p5-p30 except p19 and p20) as 
// an argument. Within your interrupt service routine call the appropriate debounce
// method debouncePinRise or debouncePinFall. 
// Requires the InterruptMask library available at: https://mbed.org/users/nleoni/code/InterruptMask/

//
// Example Code: toggles LED1 on falling edge interrupt
//#include "mbed.h"
//#include "DebounceInt2.h"
//DigitalOut myled(LED1);
//InterruptIn upPressed(p15);
//DebounceInt2 *upPressedDebounce; //debounce object to be used within interrupt for joystick up
//                                //note that pointer is defined here to have global variable
//                                //but constructor call must be within the main function.
//int main() {
//  upPressedDebounce = new DebounceInt2(p15,50000); //create DebounceInt2 object for p15 interrupt
//                                                  //with a 50000 us delay (interrupt disabled to avoid bounce)                                    
//  upPressed.fall(ISRup);            //attach joystick up click on falling edge, that is when
//                                    //user releases the joystick the action will happen
//  while(1){
//  }
//}
//void ISRup(void){
//    upPressedDebounce->debouncePinFall(); //call debounce method on pin fall, will disable interrupt for 
//                                          //specified delay
//    myled=!myled;
//}


#include "mbed.h"
#include "InterruptMask.h"

#ifndef DebounceInt2Exclude
#define DebounceInt2Exclude
#define _DEBOUNCEINTDELAYDEFAULT 50000 //useconds, this number would be about 50 ms
#define _MINDBOUNCEDELAY 50

//typedef void (*t_fhandler)(void);

class DebounceInt2 
{
public://members
private://members
    InterruptMask* pinIntMask; //Internal InterruptMask object
    unsigned int delay; //this is the debounce delay, during which the interrupt in question is masked
    Timeout debounceCallBack;
    InterruptIn* intRise;
    InterruptIn* intFall;
    bool riseAttached,fallAttached;
    FunctionPointer *riseHandler,*fallHandler;
    PinName pin;

public://methods
    DebounceInt2();     //Default constructor, required but not to be used, will yield error and exit.
    DebounceInt2(PinName pin);//Constructor with PinName, delay will remain default value
    DebounceInt2(PinName pin,unsigned int delay_us);//Constructor with PinName and specified delay in micro-seconds
    void setDelay(unsigned int); //method to modify delay for debounce in us
    void rise(FunctionPointer);     //attach interrupt handler for interrupt on rising edge
    template<typename T>
    void rise(T* tptr, void (T::*mptr)(void));
    void disableRise(void);         //rise interrupt handler for rising edge
    void enableRise(void);          //unmask rise interrupt handler for rising edge
    void fall(FunctionPointer);          //attach interrupt handler for interrupt on falling edge
    template<typename T>
    void fall(T* tptr, void (T::*mptr)(void));
    void disableFall(void);         //mask fall interrupt handler for rising edge
    void enableFall(void);          //unmask fall interrupt handler for rising edge
    
private://methods
    void debouncePinRise(void);  //method to be called within Interrupt Service Routine to debounce GPIO pin on rising edge
    void debouncePinFall(void);  //method to be called within Interrupt Service Routine to debounce GPIO pin on falling edge
    void debounceCallbackPinRise(void); //internal callback used to re-enable (unmask) interrupts on rising edge
    void debounceCallbackPinFall(void); //internal callback used to re-enable (unmask) interrupts on falling edge
    
};

#endif
