/* DebounceInt2 Library */
/* This class enables debouncing a GPIO interrupt from a specific MBED pin            */
/* It does so by using the InterruptMask library to temporarily mask the interrupt    */
/* from the specified pin for a user specified delay specified in us. After the delay */
/* time has passed the interrupt is unmasked (reenabled) and any pending interrupts   */
/* also cleared. The library gives you the flexibiity to debounce on either the       */
/* rising or falling edge interrupts.                                                 */
// Usage:
// Declare a DebounceInt2 object with a valid mbed pin (p5-p30 except p19 and p20) as 
// an argument. Within your interrupt service routine call the appropriate debounce
// method debouncePinRise or debouncePinFall. 
// Requires the InterruptMask library available at: https://mbed.org/users/nleoni/code/InterruptMask/

#include "DebounceInt2.h"

//Default constructor, required but not to be used, will yield error and exit.
    DebounceInt2::DebounceInt2(){
        exit(1);
    }
    
//Constructor with PinName, delay will remain default value
    DebounceInt2::DebounceInt2(PinName pin){
    // Inputs:
    // PinName pin: Expected Proper Mbed Pin (e.g. p12), program will exit otherwise
        this->pinIntMask = new InterruptMask(pin);
        this->setDelay(_DEBOUNCEINTDELAYDEFAULT);    
    }

//Constructor with PinName and specified delay in micro-seconds
    DebounceInt2::DebounceInt2(PinName pin,unsigned int delay_us){
    // Inputs:
    // PinName pin: Expected Proper Mbed Pin (e.g. p12), program will exit otherwise
    // unsigned int delay: Expected positive delay, will force delay to be larger
    // than minimum delay _MINDBOUNCEDELAY
        this->pinIntMask = new InterruptMask(pin);        
        this->setDelay(delay_us);
    }

//method to modify delay for debounce in us
    void DebounceInt2::setDelay(unsigned int delay_us){
    // Inputs:
    // unsigned int delay: Expected positive delay, will force delay to be larger
    // than minimum delay _MINDBOUNCEDELAY
        if(delay_us>(_MINDBOUNCEDELAY) ){
            this->delay=delay_us;
        } else { 
            this->delay=_MINDBOUNCEDELAY;
        }
    } 

//**********************************************************************************************//
// void DebounceInt2::attachRise(*fhandler){                                                    //
// Inputs: *fhandler(void), pointer to handler function which must be void *fhandler(void)                                                                               //
//**********************************************************************************************//

    void DebounceInt2::rise(FunctionPointer fhandler){     //attach interrupt handler for interrupt on rising edge
    
        if(!(this->intRise)){//only allocate the pointer and Interrupt in once
            this->intRise= new InterruptIn(this->pin);
        }
        this->riseHandler = new FunctionPointer(fhandler);
        this->intRise->rise(this,&DebounceInt2::debouncePinRise); //This attaches our handler which 
                                                                    //includes the debounce functionality                                                                
    }

    template<typename T>
    void rise(T* tptr, void (T::*mptr)(void)){
        if(!(this->intRise)){//only allocate the pointer and Interrupt in once
            this->intRise= new InterruptIn(this->pin);
        }
        this->riseHandler = new FunctionPointer(tptr,mptr);
        this->intRise->rise(this,&DebounceInt2::debouncePinRise);
    }

    
    void DebounceInt2::disableRise(void){         //rise interrupt handler for rising edge
        this->pinIntMask->maskIntR();
    }
    void DebounceInt2::enableRise(void){          //unmask rise interrupt handler for rising edge
        this->pinIntMask->unMaskIntR();    
    }

    void DebounceInt2::fall(FunctionPointer fhandler){     //attach interrupt handler for interrupt on falling edge
    
        if(!(this->intFall)){//only allocate the pointer and Interrupt in once
            this->intFall= new InterruptIn(this->pin);
        }
        this->fallHandler = new FunctionPointer(fhandler);
        this->intFall->fall(this,&DebounceInt2::debouncePinFall); //This attaches our handler which 
                                                                    //includes the debounce functionality                                                                
    
    }

    template<typename T>
    void fall(T* tptr, void (T::*mptr)(void)){
        if(!(this->Fall)){//only allocate the pointer and Interrupt in once
            this->intFall= new InterruptIn(this->pin);
        }
        this->fallHandler = new FunctionPointer(tptr,mptr);
        this->intFall->fall(this,&DebounceInt2::debouncePinRise);
    }

    void DebounceInt2::disableFall(void){         //mask fall interrupt handler for rising edge
        this->pinIntMask->maskIntF();    
    }
    void DebounceInt2::enableFall(void){          //unmask fall interrupt handler for rising edge
        this->pinIntMask->unMaskIntF();
    }


//method to be called within Interrupt Service Routine to debounce GPIO pin on rising edge
    void DebounceInt2::debouncePinRise(void){
        this->pinIntMask->maskIntR();
        this->debounceCallBack.attach_us(this,&DebounceInt2::debounceCallbackPinRise,this->delay);
        this->riseHandler->call();
    }
    
//internal callback used to re-enable (unmask) interrupts on rising edge
    void DebounceInt2::debounceCallbackPinRise(void){
        this->pinIntMask->ClrInt();
        this->pinIntMask->unMaskIntR();
    }

//method to be called within Interrupt Service Routine to debounce GPIO pin on falling edge
    void DebounceInt2::debouncePinFall(void){
        this->pinIntMask->maskIntF();
        this->debounceCallBack.attach_us(this,&DebounceInt2::debounceCallbackPinFall,this->delay);
        this->fallHandler->call();
    }
    
//internal callback used to re-enable (unmask) interrupts on falling edge
    void DebounceInt2::debounceCallbackPinFall(void){
        this->pinIntMask->ClrInt();
        this->pinIntMask->unMaskIntF();
    }