/* DebounceInt Library */
/* This class enables debouncing a GPIO interrupt from a specific MBED pin            */
/* It does so by using the InterruptMask library to temporarily mask the interrupt    */
/* from the specified pin for a user specified delay specified in us. After the delay */
/* time has passed the interrupt is unmasked (reenabled) and any pending interrupts   */
/* also cleared. The library gives you the flexibiity to debounce on either the       */
/* rising or falling edge interrupts.                                                 */
// Usage:
// Declare a DebounceInt object with a valid mbed pin (p5-p30 except p19 and p20) as 
// an argument. Within your interrupt service routine call the appropriate debounce
// method debouncePinRise or debouncePinFall. 
// Requires the InterruptMask library available at: https://mbed.org/users/nleoni/code/InterruptMask/

#include "DebounceInt.h"

//Default constructor, required but not to be used, will yield error and exit.
    DebounceInt::DebounceInt(){
        exit(1);
    }
    
//Constructor with PinName, delay will remain default value
    DebounceInt::DebounceInt(PinName pin){
    // Inputs:
    // PinName pin: Expected Proper Mbed Pin (e.g. p12), program will exit otherwise
        this->pinIntMask = new InterruptMask(pin);
        this->setDelay(_DEBOUNCEINTDELAYDEFAULT);    
    }

//Constructor with PinName and specified delay in micro-seconds
    DebounceInt::DebounceInt(PinName pin,unsigned int delay_us){
    // Inputs:
    // PinName pin: Expected Proper Mbed Pin (e.g. p12), program will exit otherwise
    // unsigned int delay: Expected positive delay, will force delay to be larger
    // than minimum delay _MINDBOUNCEDELAY
        this->pinIntMask = new InterruptMask(pin);        
        this->setDelay(delay_us);
    }

//method to modify delay for debounce in us
    void DebounceInt::setDelay(unsigned int delay_us){
    // Inputs:
    // unsigned int delay: Expected positive delay, will force delay to be larger
    // than minimum delay _MINDBOUNCEDELAY
        if(delay_us>(_MINDBOUNCEDELAY) ){
            this->delay=delay_us;
        } else { 
            this->delay=_MINDBOUNCEDELAY;
        }
    } 

//method to be called within Interrupt Service Routine to debounce GPIO pin on rising edge
    void DebounceInt::debouncePinRise(void){
        this->pinIntMask->maskIntR();
        this->debounceCallBack.attach_us(this,&DebounceInt::debounceCallbackPinRise,this->delay);
    }
    
//internal callback used to re-enable (unmask) interrupts on rising edge
    void DebounceInt::debounceCallbackPinRise(void){
        this->pinIntMask->ClrInt();
        this->pinIntMask->unMaskIntR();
    }

//method to be called within Interrupt Service Routine to debounce GPIO pin on falling edge
    void DebounceInt::debouncePinFall(void){
        this->pinIntMask->maskIntF();
        this->debounceCallBack.attach_us(this,&DebounceInt::debounceCallbackPinFall,this->delay);
    }
    
//internal callback used to re-enable (unmask) interrupts on falling edge
    void DebounceInt::debounceCallbackPinFall(void){
        this->pinIntMask->ClrInt();
        this->pinIntMask->unMaskIntF();
    }