/* DebounceInt Library */
/* This class enables debouncing a GPIO interrupt from a specific MBED pin            */
/* It does so by using the InterruptMask library to temporarily mask the interrupt    */
/* from the specified pin for a user specified delay specified in us. After the delay */
/* time has passed the interrupt is unmasked (reenabled) and any pending interrupts   */
/* also cleared. The library gives you the flexibiity to debounce on either the       */
/* rising or falling edge interrupts.                                                 */
// Usage:
// Declare a DebounceInt object with a valid mbed pin (p5-p30 except p19 and p20) as 
// an argument. Within your interrupt service routine call the appropriate debounce
// method debouncePinRise or debouncePinFall. 
// Requires the InterruptMask library available at: https://mbed.org/users/nleoni/code/InterruptMask/

//
// Example Code: toggles LED1 on falling edge interrupt
//#include "mbed.h"
//#include "DebounceInt.h"
//DigitalOut myled(LED1);
//InterruptIn upPressed(p15);
//DebounceInt *upPressedDebounce; //debounce object to be used within interrupt for joystick up
//                                //note that pointer is defined here to have global variable
//                                //but constructor call must be within the main function.
//int main() {
//  upPressedDebounce = new DebounceInt(p15,50000); //create DebounceInt object for p15 interrupt
//                                                  //with a 50000 us delay (interrupt disabled to avoid bounce)                                    
//  upPressed.fall(ISRup);            //attach joystick up click on falling edge, that is when
//                                    //user releases the joystick the action will happen
//  while(1){
//  }
//}
//void ISRup(void){
//    upPressedDebounce->debouncePinFall(); //call debounce method on pin fall, will disable interrupt for 
//                                          //specified delay
//    myled=!myled;
//}


#include "mbed.h"
#include "InterruptMask.h"

#ifndef DEBOUNCEINT
#define DEBOUNCEINT
#define _DEBOUNCEINTDELAYDEFAULT 50000 //useconds, this number would be about 50 ms
#define _MINDBOUNCEDELAY 5
class DebounceInt{
public://members
private://members
    InterruptMask* pinIntMask; //Internal InterruptMask object
    unsigned int delay; //this is the debounce delay, during which the interrupt in question is masked
    Timeout debounceCallBack;

public://methods
    DebounceInt();     //Default constructor, required but not to be used, will yield error and exit.
    DebounceInt(PinName pin);//Constructor with PinName, delay will remain default value
    DebounceInt(PinName pin,unsigned int delay_us);//Constructor with PinName and specified delay in micro-seconds
    void setDelay(unsigned int); //method to modify delay for debounce in us
    void debouncePinRise(void);  //method to be called within Interrupt Service Routine to debounce GPIO pin on rising edge
    void debouncePinFall(void);  //method to be called within Interrupt Service Routine to debounce GPIO pin on falling edge
    
private://methods
    void debounceCallbackPinRise(void); //internal callback used to re-enable (unmask) interrupts on rising edge
    void debounceCallbackPinFall(void); //internal callback used to re-enable (unmask) interrupts on falling edge
    
};

#endif
