#ifndef __META_H
#define __META_H

#include "includes.h"
#include "core.h"
#include "sensors.h"

class Modulator {
public:
    Modulator(Inverter *inverter) {_inverter = inverter;}
    virtual void Update(float va, float vb) = 0;
protected:
    Inverter* _inverter;
};

class SinusoidalModulator: public Modulator {
public:
    SinusoidalModulator(Inverter *inverter):Modulator(inverter) {}
    virtual void Update(float va, float vb);
private:
    static float LutSin(float theta);
    static float LutCos(float theta);
};

class PidController {
public:
    PidController(float ki, float kp, float kd, float out_max, float out_min);
    float Update(float ref, float in);
private:
    float _ki, _kp, _kd;
    float _last_in, _integral;
    float _out_max, _out_min;
}; 

class ReferenceSynthesizer {
public:
    ReferenceSynthesizer(float max_phase_current) {_max_phase_current = max_phase_current;}
    virtual void GetReference(float angle, float throttle, float *ref_d, float *ref_q) {*ref_d = 0; *ref_q = 0;}
protected:
    static float LutSin(float theta);
    static float LutCos(float theta);
protected:
    float _max_phase_current;
};

class SynchronousReferenceSynthesizer : public ReferenceSynthesizer {
public:
    SynchronousReferenceSynthesizer(float max_phase_current):ReferenceSynthesizer(max_phase_current) {}
    virtual void GetReference(float angle, float throttle, float *ref_d, float *ref_q);
};

class StatusUpdater {
public:
    StatusUpdater(Inverter *inverter, Motor *motor, User *user);
    void Config(int fast_sample_rate, int med_sample_rate, int slow_sample_rate);
    void Start();
private:
    static void time_upd_isr();
    static float LutSin(float theta);
    static float LutCos(float theta);
private:
    Inverter *_inverter;
    Motor *_motor;
    User *_user;
    int _fast_sample_rate;
    int _med_sample_rate;
    int _slow_sample_rate;
    
    static unsigned long _time;
    Ticker _time_ticker;
};

class LoopDriver {
public:
    LoopDriver(Inverter *inverter, Motor *motor, User *user, PidController *pid_d, PidController *pid_q,  
               Modulator *modulator, float max_phase_current, int update_frequency);
    void Start();
private:
    static void Clarke(float a, float b, float *alpha, float *beta);
    static void Parke(float alpha, float beta, float theta, float *d, float *q);
    static void InverseParke(float d, float q, float theta, float *alpha, float *beta);
private:
    static void update();
private:
    static float LutSin(float theta);
    static float LutCos(float theta);
private:
    static Inverter *_inverter;
    static Motor *_motor;
    static User *_user;
    static PidController *_pid_d, *_pid_q;
    static ReferenceSynthesizer *_reference;
    static Modulator *_modulator;
    
    Ticker _upd_ticker;
    float _max_phase_current;
    int _update_frequency;
    static int _blinky_toggle;
};

#endif

