/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef TCPSOCKET_H
#define TCPSOCKET_H

#include "Socket/Socket.h"
#include "Socket/Endpoint.h"

/**
This is a C++ abstraction for TCP networking sockets.
*/
class TCPSocketConnection : public Socket, Endpoint {
    friend class TCPSocketServer;
    
public:
    /** Instantiate a TCP Socket.
    */
    TCPSocketConnection();
    
    ~TCPSocketConnection();
    
    /** Connect the TCP Socket to the following host.
    \param host The host to connect to. It can either be an IP Address or a hostname that will be resolved with DNS.
    \param port The host's port to connect to.
    \return 0 on success, -1 on failure.
    */
    int connect(const char* host, const int port);
    
    /** Send data to the remote host.
    \param data The buffer to send to the host.
    \param length The length of the buffer to send.
    \param timeout The maximum amount of time in ms to wait while trying to send the buffer.
    \return the number of written bytes on success (>=0) or -1 on failure
     */
    int send(char* data, int length, int timeout_ms=0);
    
    /** Send data to the remote host.
    \param data The buffer to send to the host.
    \param length The length of the buffer to send.
    \param timeout The maximum amount of time in ms to wait while trying to send the buffer.
    \return the number of written bytes on success (>=0) or -1 on failure
    */
    int send_all(char* data, int length, int timeout_ms=0);
    
    /** Receive data from the remote host.
    \param data The buffer in which to store the data received from the host.
    \param length The maximum length of the buffer.
    \param timeout The maximum amount of time in ms to wait while trying to receive data.
    \return the number of received bytes on success (>=0) or -1 on failure
     */
    int receive(char* data, int length, int timeout_ms=0);
    
    /** Receive data from the remote host.
    \param data The buffer in which to store the data received from the host.
    \param length The maximum length of the buffer.
    \param timeout The maximum amount of time in ms to wait while trying to receive data.
    \return the number of received bytes on success (>=0) or -1 on failure
    */
    int receive_all(char* data, int length, int timeout_ms=0);

private:
    bool _closedByRemoteHost;

};

#endif
