/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "TCPSocketConnection.h"
#include <cstring>

using std::memset;
using std::memcpy;

TCPSocketConnection::TCPSocketConnection() :
        _closedByRemoteHost(false) {
}

int TCPSocketConnection::connect(const char* host, const int port) {
    if (init_socket(SOCK_STREAM) < 0)
        return -1;
    
    if (set_address(host, port) != 0)
        return -1;
    
    if (lwip_connect(_sock_fd, (const struct sockaddr *) &_remoteHost, sizeof(_remoteHost)) < 0) {
        close();
        return -1;
    }
    
    return 0;
}

int TCPSocketConnection::send(char* data, int length, int timeout_ms) {
    if ((_sock_fd < 0) || _closedByRemoteHost)
        return -1;
    
    TimeInterval timeout(timeout_ms);
    if (wait_writable(timeout) != 0)
        return -1;
    
    int n = lwip_send(_sock_fd, data, length, 0);
    _closedByRemoteHost = (n == 0);
    
    return n;
}

// -1 if unsuccessful, else number of bytes written
int TCPSocketConnection::send_all(char* data, int length, int timeout_ms) {
    if ((_sock_fd < 0) || _closedByRemoteHost)
        return -1;
    
    size_t writtenLen = 0;
    TimeInterval timeout(timeout_ms);
    while (writtenLen < length) {
        // Wait for socket to be writeable
        if (wait_writable(timeout) != 0)
            return writtenLen; //Timeout -- FIXME should we return -1 or writtenLength ?
        
        int ret = lwip_send(_sock_fd, data + writtenLen, length - writtenLen, 0);
        if (ret > 0) {
            writtenLen += ret;
            continue;
        } else if (ret == 0) {
            _closedByRemoteHost = true;
            return writtenLen; //Connection was closed by remote host -- FIXME how do we signal that the connection was closed ?
        } else {
            return -1; //Connnection error
        }
    }
    
    return writtenLen;
}

int TCPSocketConnection::receive(char* data, int length, int timeout_ms) {
    if ((_sock_fd < 0) || _closedByRemoteHost)
        return -1;
    
    TimeInterval timeout(timeout_ms);
    if (wait_readable(timeout) != 0)
        return -1;
    
    int n = lwip_recv(_sock_fd, data, length, 0);
    _closedByRemoteHost = (n == 0);
    
    return n;
}

// -1 if unsuccessful, else number of bytes received
int TCPSocketConnection::receive_all(char* data, int length, int timeout_ms) {
    if ((_sock_fd < 0) || _closedByRemoteHost)
        return -1;
    
    size_t readLen = 0;
    TimeInterval timeout(timeout_ms);
    while (readLen < length) {
        //Wait for socket to be readable
        if (wait_readable(timeout) != 0)
            return readLen; //Timeout -- FIXME should we return -1 or writtenLength ?
        
        int ret = lwip_recv(_sock_fd, data + readLen, length - readLen, 0);
        if (ret > 0) {
            readLen += ret;
        } else if (ret == 0) {
            _closedByRemoteHost = true;
            return readLen; //Connection was closed by remote host -- FIXME how do we signal that the connection was closed ?
        } else {
            return -1; //Connnection error
        }
    }
    return readLen;
}

TCPSocketConnection::~TCPSocketConnection() {
    close(); //Don't want to leak 
}
