/*
 * np_system.c
 *
 *  Created on: July 14, 2016
 *      Author: Alan.Lin
 *
 *  Copyright: NexPack Ltd.
 */

#include "np_config.h"
#include "np_driver_timer.h"
#include "np_driver_spi.h"
#include "np_app_spi.h"
#include "np_app_ncn_interface.h"
#include "spis.h"
#include "np_system.h"
#include "np_command.h"

extern const gpio_cfg_t gpio_req_spi[];

/* Description: delay function
 * t_ms: the time for delay.MS as the unit
 *
 */
void delay_ms(uint32_t t_ms)
{
    while(t_ms) {
        SYS_SysTick_Delay(96000);
        t_ms--;
    }
}

/*
 * Description: disable all interrupt used on MDK
 * Parameter  : null
 * Return     : null
 */
void np_system_disable_interrupt(void)
{
    NVIC_DisableIRQ(TMR0_0_IRQn);
    NVIC_DisableIRQ(SPIS_IRQn);
}

/*
 * Description: Setting the version of MDK
 * Parameter: mdk version
 * Return: null
 */
void np_function_set_mdk_version(uint8_t HV, uint8_t MV, uint8_t LV)
{
    mdk_mdk_version[0] = HV;
    mdk_mdk_version[1] = MV;
    mdk_mdk_version[2] = LV;
}

/*
 * Description: MDK initial
 * Parameter  : null
 * Return     : null
 */
void np_mdk_setup(void)
{
    int t_reback = E_BUSY;

    GPIO_Config(&gpio_req_spi[0]);	//Input
    //ignore clock initial
    while(t_reback != E_NO_ERROR) {
        t_reback = driver_spi_slave_initial();
        delay_ms(10);
    }

    Timer0_initial();
    np_function_set_post_address(RAM_SOURCE_ADDR);

    spi_status = SPI_FREE;
    my_node = RAM_MY_NODE;
}

/*
 * Description: system initial,include app initial and mdk initial
 * Parameter  : null
 * Return     : null
 */
void np_system_initial(app_function np_app_setup)
{
    uint8_t answer = 0;

    np_function_set_mdk_version(1,0,7);
    np_mdk_setup();
    np_app_setup();
    np_function_ncn_interface_post_message(RSP_INTO_APP, &answer, 1);
    flag_jump_bsl = 0;
}

/*
 * Description: mdk initialization
 * Parameter  : null
 * Return     : null
 */
void np_sys_init()
{
    np_function_set_mdk_version(1,0,7);
    np_mdk_setup();
#ifdef DBG_MSG            
      		printf("NP System Initialized\n\r");  
#endif            
}

/*
 * Description: signal when system is ready
 * Parameter  : null
 * Return     : null
 */
void np_sys_start()
{
    uint8_t answer = 0;
    np_function_ncn_interface_post_message(RSP_INTO_APP, &answer, 1);
    flag_jump_bsl = 0;
#ifdef DBG_MSG            
      		printf("NP System Started\n\r");  
#endif            
}

/*
 * Description: function to check and jump to boot loader
 * Parameter  : null
 * Return     : null
 */
#ifdef TOOLCHAIN_ARM_STD
register int _r0 __asm("r0");
register int _pc __asm("pc");
#endif

void np_sys_bsl_chk()
{
    if( flag_jump_bsl > 0x00 ) {
        flag_jump_bsl = 0x00;
        np_system_disable_interrupt();
#ifdef DBG_MSG            
      		printf("Entering Bootloader\n\r");  
#endif            
#ifdef TOOLCHAIN_ARM_STD
        __asm {
            mov _r0, #0x20000008
            ldr _pc, [_r0]
        }
    }
    __asm {
        mov _r0, _r0
        mov _r0, _r0
    }
#else
        __asm volatile(
            " LDR R0, =0x20000008	\n"
            " LDR PC, [R0]			\n"
        );
    }
    __asm volatile("nop \n");
    __asm volatile("nop \n");
#endif
}

/*
 * Description: system loop,software will run the function forever when software is on app mode.
 * Parameter  : null
 * Return     : null
 */
void np_system_loop(app_function np_api_loop)
{
    while(1) {
        np_api_loop();
        //jumping to the bootloader
        if( flag_jump_bsl > 0x00 ) {
            flag_jump_bsl = 0x00;
            np_system_disable_interrupt();
            
#ifdef DBG_MSG            
      		printf("Entering NP Bootloader\n\r");  
#endif            
#ifdef TOOLCHAIN_ARM_STD
            __asm {
                mov _r0, #0x20000008
                ldr _pc, [_r0]
            }
        }
        __asm {
            mov _r0, _r0
            mov _r0, _r0
        }
#else
            __asm volatile(
                " LDR R0, =0x20000008	\n"
                " LDR PC, [R0]			\n"
            );
        }
        __asm volatile("nop \n");
        __asm volatile("nop \n");
#endif
    }
}

