/* mbed Sensiron SHT7x temperature and humidity sensor library
*
* Sensiron data at http://www.sensirion.com/en/01_humidity_sensors/06_humidity_sensor_sht75.htm
*
* Copyright (c) Ian Molesworth October 2010
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to use
* copy or modify the software for private or non-commercial purposes only. 
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/


#ifndef SHT75_H
#define SHT75_H

#include "mbed.h"

const float C1= -4.0;              // for 12 Bit humi
const float C2= +0.0405;           // for 12 Bit humi
const float C3= -0.0000028;        // for 12 Bit hum
const float TL= -39.61;             // 3v 14 bit 
const float T1= +0.01;             // for 14 Bit @ 5V
const float T2= +0.00008;          // for 14 Bit @ 5V

/** SHT7X driver class
*
* Example:
* @code
* //initialise the device read temperature ticks, read humidity ticks and then
* // calculate the liniarised humidity value.
* #include "mbed.h"
* #include "sht7X.h"
* 
* SHT75 sht(p12, p11); 
* Serial pc(USBTX, USBRX); 
* 
* int main() 
*     {
*     float temperature;                    // temperature -40 to 120 deg C
*     float humidity;                       // relative humidity 1% to 100%
*     float humi_f,rh_lin,rh_true;           // working registers for Illustration purposes
*     int t;                                // temporary store for the temp ticks
*     int h;                                // temp store for the humidity ticks
*
*     pc.baud(115200);
*
*     while(1)
*         {
*         sht.readTempTicks(&t); 
*         temperature = ((float)(t) * 0.01) - 39.61;
*
*         sht.readHumidityTicks(&h); 
*         humi_f = (float)(h);
*         rh_lin = C3 * humi_f * humi_f + C2 * humi_f + C1;
*         rh_true=(((temperature/100)-25)*(T1+T2*humi_f)+rh_lin);
*         if(rh_true>100)rh_true=100;                             //cut if the value is outside
*         if(rh_true<1)rh_true=1;                                 //the physical possible range
*         humidity = rh_true;                   
*         pc.printf("Temp: %2.2f RH %2.2f\n\r",temperature, humidity);         
*         }
*     }
* @endcode
*/

class SHT75 
    {
    public:
     /**
      * @var temperature float to hold the temp
      */
        float temperature;
      /**
      * @var humidity float to hold the calculated humidity
      */

        float humidity;
    
     /** Create an SHT object connected to the specified Digital pins
      *
      * @param pclock digital pin to use as clock
      * @param pdata digital pin to use as data bus ( bidirectional ) 
      */
        SHT75(PinName pclock, PinName pdata): _clock(pclock), _data(pdata) {};
     /** read the temperature ticks value 14 bit resolution
      *
      * @param int *temp pointer to an integer to hold the tick value
      * @returns boolean true if read acknowledges
      */
        bool readTempTicks(int* temp);
     /** read the humidity ticks value 12 bit resolution
      *
      * @param int *temp pointer to an integer to hold the tick value
      * @returns boolean true if read acknowledges
      */
        bool readHumidityTicks(int* temp);
     /** start up reset
      *
      * call to resync or abort current operation.
      * worth calling every now and then to make sure your system is not hung.
      */
        void reset(void);
        void softReset(void);
        int  readStatus(void);
  
    private:  
        DigitalInOut _data;
        DigitalOut _clock;
        
        void start(void);
        int read(char);
        bool write(char);
    };

#endif
