/* ESP8266 Example
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "ESP8266.h"

ESP8266::ESP8266(PinName tx, PinName rx, int locOutPort, int locInPort, bool debug)
    : _serial(tx, rx, 1024), _parser(_serial),
      localOutPort(locOutPort), localInPort(locInPort)
{
    _serial.baud(115200);
    _parser.debugOn(debug);
}

bool ESP8266::startup(int mode)
{
    if(mode < 1 || mode > 3) {
        return false;
    }

    return reset()
           && _parser.send("AT+CWMODE_CUR=%d", mode)
           && _parser.recv("OK")
           && _parser.send("AT+CIPMUX=1")
           && _parser.recv("OK")
           && _parser.send("AT+CIPDINFO=1")
           && _parser.recv("OK");
}

bool ESP8266::reset(void)
{
    for (int i = 0; i < 2; i++) {
        if (_parser.send("AT+RST")
                && _parser.recv("OK\r\nready")) {
            return true;
        }
    }

    return false;
}

bool ESP8266::connect(const char *ap, const char *passPhrase)
{
    setTimeout(20000);
    bool b = _parser.send("AT+CWJAP_CUR=\"%s\",\"%s\"", ap, passPhrase) && _parser.recv("OK");
    setTimeout(8000);
    return b;
}

bool ESP8266::disconnect(void)
{
    return _parser.send("AT+CWQAP") && _parser.recv("OK");
}

char* ESP8266::getIPAddress(void)
{
    if (!(_parser.send("AT+CIFSR")
            && _parser.recv("+CIFSR:STAIP,\"%[^\"]\"", _ip_buffer)
            && _parser.recv("OK"))) {
        return 0;
    }

    return _ip_buffer;
}

const char* ESP8266::getMACAddress(void)
{
    if (!(_parser.send("AT+CIFSR")
            && _parser.recv("+CIFSR:STAMAC,\"%[^\"]\"", _mac_buffer)
            && _parser.recv("OK"))) {
        return 0;
    }

    return _mac_buffer;
}

bool ESP8266::isConnected(void)
{
    return getIPAddress() != 0;
}

bool ESP8266::open(int linkId, int localPort)
{
    return _parser.send("AT+CIPSTART=%d,\"%s\",\"%s\",%d,%d,%d", linkId, "UDP","192.168.0.0",49000, localPort, 2)
           && _parser.recv("OK");
}

bool ESP8266::send(const void* data, uint32_t amount, const char* addr, int port)
{
    //May take a second try if device is busy
    for (unsigned i = 0; i < 4; i++) {
        if (_parser.send("AT+CIPSEND=%d,%d,\"%s\",%d", 1, amount, addr, port)
                && _parser.recv(">")
                && (_parser.write((char*)data, (int)amount) >= 0)
                && _parser.recv("SEND OK")) {
            return true;
        }
    }

    return false;
}

int32_t ESP8266::recv(void *data, uint32_t amount, char* IP, int* port)
{
    uint32_t linkId;
    uint32_t recv_amount;

    if (!(_parser.recv("+IPD,%d,%d,%[^\,],%d:", &linkId, &recv_amount, IP, port)
            && linkId == 2
            && recv_amount <= amount
            && _parser.read((char*)data, recv_amount))) {
        return -1;
    }

    return recv_amount;
}

bool ESP8266::close(int id)
{
    //May take a second try if device is busy
    for (unsigned i = 0; i < 2; i++) {
        if (_parser.send("AT+CIPCLOSE=%d", id)
                && _parser.recv("OK")) {
            return true;
        }
    }

    return false;
}

void ESP8266::setTimeout(uint32_t timeout_ms)
{
    _parser.setTimeout(timeout_ms);
}

bool ESP8266::readable()
{
    return _serial.readable();
}

bool ESP8266::writeable()
{
    return _serial.writeable();
}

bool ESP8266::setupAP()
{
    return _parser.send("AT+CWSAP_CUR=\"RobotClapeyronAP\",\"passpass123\",5,3")
           && _parser.recv("OK")
           && open(1,localOutPort)
           && open(2,localInPort);
}

