#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>
#include "DHT.h"

// these options must match the settings on your Conduit
// TTN Keys
static const uint8_t netowork_session_key_array[] = {0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C};
static const uint8_t data_session_key_array[] = {0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C};
// uncomment the following lines and edit their values to match your configuration
static const uint8_t network_address_array[] = {0x02, 0x01, 0xBA, 0x16}; // use yours based on http://thethingsnetwork.org/wiki/AddressSpace
static std::vector<uint8_t> netowork_session_key (netowork_session_key_array, netowork_session_key_array + sizeof(netowork_session_key_array) / sizeof(uint8_t));
static std::vector<uint8_t> data_session_key (data_session_key_array, data_session_key_array + sizeof(data_session_key_array) / sizeof(uint8_t));
static std::vector<uint8_t> network_address (network_address_array, network_address_array + sizeof(network_address_array) / sizeof(uint8_t));
static uint8_t config_frequency_sub_band = 4;


int main() {
    int32_t ret;
    mDot* dot;
    std::vector<uint8_t> data;
    DHT sensor(PA_11,DHT11); // Use the DHT11 sensor connected to Vcc GND and on pin D7 UKD2
    std::string data_str = "Hello Nico!";


    // get a mDot handle
    dot = mDot::getInstance();

    dot->resetConfig();

    dot->setLogLevel(mts::MTSLog::DEBUG_LEVEL);

    // too lazzy to check all errors
    dot->setJoinMode(mDot::MANUAL);
    dot->setPublicNetwork(true);
    dot->setFrequencySubBand(config_frequency_sub_band);
    dot->setNetworkSessionKey(netowork_session_key);
    dot->setDataSessionKey(data_session_key);
    dot->setNetworkAddress(network_address);
    
    
    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, spreading factors 7 - 10 are available
    // in the 868 (EU) frequency band, spreading factors 7 - 12 are available
    logInfo("setting TX spreading factor");
    if ((ret = dot->setTxDataRate(mDot::SF_10)) != mDot::MDOT_OK) {
        logError("failed to set TX datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    
    
    // request not to receive confirmation of packets from the gateway since TTN doesn't support it
    logInfo("enabling ACKs");
    if ((ret = dot->setAck(0)) != mDot::MDOT_OK) {
        logError("failed to enable ACKs %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // save this configuration to the mDot's NVM
    logInfo("saving config");
    if (! dot->saveConfig()) {
        logError("failed to save configuration");
    }
    //*******************************************
    // end of configuration
    //*******************************************

    // attempt to join the network
    logInfo("joining network");
    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        osDelay(std::max((uint32_t)1000, (uint32_t)dot->getNextTxMs()));
    }

    int sensor_error = 0;
    float temp = 0;
    float humid = 0;
    int update_period = 5000;
    std::string separator_str = ",";
    char string_buffer[64];

    while (true) {
        data.clear();

        sensor_error = sensor.readData();
        if (sensor_error !=0)
            printf("\r\nErr %i \n",sensor_error);
        while ( sensor_error != 0 ) {
            sensor_error = sensor.readData();
            osDelay(100);
        }
        // TEMP
        temp = sensor.ReadTemperature(CELCIUS);
        sprintf(string_buffer, "%4.2f", temp);
        for (int i = 0; i<strlen(string_buffer); i++)
        {
            data.push_back(((char*)string_buffer)[i]);
        }
        // SEPARATOR
        data.push_back((uint8_t) *separator_str.begin());
        // HUMIDITY
        humid = sensor.ReadHumidity();
        sprintf(string_buffer, "%4.2f", humid);
        for (int i = 0; i<strlen(string_buffer); i++)
        {
            data.push_back(((char*)string_buffer)[i]);
        }

        logInfo("Temperature is %4.2f C",temp);
        logInfo("Temperature is %4.2f F",sensor.ReadTemperature(FARENHEIT));
        logInfo("Temperature is %4.2f K",sensor.ReadTemperature(KELVIN));
        logInfo("Humidity is %4.2f",humid);
        logInfo("Dew point is %4.2f",sensor.CalcdewPoint(sensor.ReadTemperature(CELCIUS), sensor.ReadHumidity()));
        logInfo("Dew point (fast) is %4.2f",sensor.CalcdewPointFast(sensor.ReadTemperature(CELCIUS), sensor.ReadHumidity()));
        
        logDebug("Sending LoRa message, length: %d", data.size());
        logDebug("sending data: ");
        for(int i = 0; i < data.size(); i++)
        {
            printf("%c", data[i]);
        }
        printf("\n");
        if ((ret = dot->send(data)) != mDot::MDOT_OK)
        {
            logError("failed to send", ret, mDot::getReturnCodeString(ret).c_str());
        } else
        {
            logInfo("successfully sent data to gateway");
            data.clear();
            if ((ret = dot->recv(data)) != mDot::MDOT_OK)  {
                logError("failed to rec", ret, mDot::getReturnCodeString(ret).c_str());
            } else {
                logDebug("successfully recd data from gateway");
                logDebug("recv data size: %i\n",data.size());
                logDebug("recv data: ");
                for(int i = 0;i < data.size();i++) {
                    printf("%c", data[i]);
                }
                printf("\n");
            }
        }
        osDelay(std::max((uint32_t)update_period, (uint32_t)dot->getNextTxMs()));
    }

    return 0;
}
