/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef USBHOSTSERIAL_H
#define USBHOSTSERIAL_H

#include "USBHost.h"
#include "Stream.h"
#include "MtxCircBuffer.h"

enum IrqType {
    RxIrq,
    TxIrq
};

class USBHostSerial: public Stream {
public:
    /*
    * Constructor
    */
    USBHostSerial();

    /*
    * Check if a serial port device is connected
    *
    * @returns true if a serial device is connected
    */
    bool connected();

    /*
    * _getc overriden to provide scanf capabilities
    */
    virtual int _getc();

    /*
    * _putc overriden to provide printf capabilities
    */
    virtual int _putc(int c);
    
    /**
    * Check the number of bytes available.
    *
    * @returns the number of bytes available
    */
    uint8_t available(); 

    /**
     *  Attach a member function to call when a packet is received.
     *
     *  @param tptr pointer to the object to call the member function on
     *  @param mptr pointer to the member function to be called
     */
    template<typename T>
    void attach(T* tptr, void (T::*mptr)(void), IrqType irq = RxIrq) {
        if ((mptr != NULL) && (tptr != NULL)) {
            if (irq == RxIrq) {
                rx.attach(tptr, mptr);
            } else {
                tx.attach(tptr, mptr);
            }
        }
    }

    /**
     * Attach a callback called when a packet is received
     *
     * @param fptr function pointer
     */
    void attach(void (*fn)(void), IrqType irq = RxIrq) {
        if (fn != NULL) {
            if (irq == RxIrq) {
                rx.attach(fn);
            } else {
                tx.attach(fn);
            }
        }
    }
    

private:
    USBHost * host;
    USBDeviceConnected * dev;
    Endpoint * bulk_in;
    Endpoint * bulk_out;

    bool dev_connected;

    void init();

    MtxCircBuffer<uint8_t, 64> circ_buf;

    uint8_t buf[20];

    void rxHandler();
    void txHandler();
    FunctionPointer rx;
    FunctionPointer tx;

};

#endif
