
/*
Copyright (C) 2012 Vodafone Limited.

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to do
so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

#define __DEBUG__ 4 //Maximum verbosity
#ifndef __MODULE__
#endif

#define MY_PHONE_NUMBER "+447717275049"     // Nicholas' Number
#define BACKUP_NUMBER "+447825608771"       // Ashleys Number

#include "mbed.h"
#include "rtos.h"
#include "VodafoneUSBModem.h"

#include <string>

extern "C" void HardFault_Handler() {
    error("Hard Fault!\n");
}

DigitalOut  led1(LED1);                     // LED 1 is used to show the state of the door. ON is door open, and OFF is door close.
DigitalOut  led2(LED2);                     // LED 2 is used to indicate that the main process is still running.
DigitalOut  led3(LED3);                     // LED 3 is use to indicate if the relay is being switched on to open the door
DigitalOut  led4(LED4);                     // LED 4 indicate the state of the ALARM - If it's on the door is armed.
DigitalOut  doorContactorRelay(p5);         // create a digital pin object to control the door contactor relay
DigitalIn   doorProximitySwitch(p6);        // create a digital pin object to sense door position proximity switch, this works in reverse
                                            // the switch senses positive when the door is closed - need to be sure when it's closed! :-)
bool        openTheDoor = false;            // create a state variable which is set to indicate the mbed should open the door
                                            // this variable will be used hold the latch open until it knows the door has been open.
bool        detectRegState = true;          // this boolean variable will force the while loop to detect reg state - if the connection manager
                                            // is not registered it will stay in this 'detec' phase for ever.  
bool        alarmActive = false;            // this alarm flag will only be true if the door is open - but no signal has been detected 
                                            // to open the door. in other words there is a break in!                                          
bool        closeTheDoor = false;           // this flag is used when the door is in an open legal state. once it's closed after a legal state
                                            // it will set the alarm to active. 


int read_digital(void) {
    DigitalIn mydigital(p6);
    return(mydigital.read());
}


// http://mbed.org/forum/mbed/topic/508/
class Watchdog 
{
    public:
    // Load timeout value in watchdog timer and enable
        void setTimeOut(float s) 
        {
            LPC_WDT->WDCLKSEL = 0x1;                // Set CLK src to PCLK
            uint32_t clk = SystemCoreClock / 16;    // WD has a fixed /4 prescaler, PCLK default is /4
            LPC_WDT->WDTC = s * (float)clk;
            LPC_WDT->WDMOD = 0x3;                   // Enabled and Reset
            kick();
        }
    // "kick" or "feed" the dog - reset the watchdog timer
    // by writing this required bit pattern
        void kick() 
        {
            LPC_WDT->WDFEED = 0xAA;
            LPC_WDT->WDFEED = 0x55;
        }
};
 

// Setup the watchdog timer
Watchdog watchDogTimer;



void doorController(void const*) {

  DBG("Hello world and Vodafone Connection Manager test program!");
  DBG("Waiting 30 seconds for modem to settle down");
  Thread::wait(10000);
  VodafoneUSBModem connectionManager;                     // create a connection manager object

  char num[17];                                           // create a buffer to hold the telephone number in
  char msg[160];                                          // create a buffer to hold the text message in
  size_t count;
  LinkMonitor::REGISTRATION_STATE regState = LinkMonitor::REGISTRATION_STATE_UNKNOWN;       // an enum to hold the registration state of the connection manager
  LinkMonitor::BEARER bearer = LinkMonitor::BEARER_UNKNOWN;                                 // an enum to hold the type of bearer established by the connection manager
  int rssi = -1000;                                                                         // a signal strength indicator variable - set to a very default low value.

  // this is a fudge to let the modem settle down - we wait for 30 seconds
  
  DBG("Wait over - initialising the modem manager");
  
  
  
  int ret = connectionManager.getLinkState(&rssi, &regState, &bearer);
  
  // LinkMonitor::REGISTRATION_STATE regState = LinkMonitor::REGISTRATION_STATE_HOME_NETWORK;

  // before we do anything we need to make sure we have a connection to a network.
  // so let us check that first!
  
  
  while(detectRegState)
  {
      if (detectRegState ==1)
      {
        if(rssi==-1000) 
        { 
            DBG("Checking signal strength - RSSI: Error.");
        } 
        else 
        { 
            DBG("Signal strength is: RSSI: %d",rssi);
        }
            
        switch(regState) 
        {
            case LinkMonitor::REGISTRATION_STATE_UNKNOWN:
                DBG("regState: UNKNOWN. Failing.");
                break;
            case LinkMonitor::REGISTRATION_STATE_REGISTERING:
                DBG("regState: REGISTERING");
                break;
            case LinkMonitor::REGISTRATION_STATE_DENIED:
                DBG("regState: DENIED");
                break;
            case LinkMonitor::REGISTRATION_STATE_NO_SIGNAL:
                DBG("regState: NO SIGNAL");
                break;
            case HARDWARE_NO_RESPONSE:
                DBG("Looks like there is no modem inserted, or power to the modem module has failed?");
                break;
            case LinkMonitor::REGISTRATION_STATE_HOME_NETWORK:
                detectRegState = false;
                DBG("regState: HOME NETWORK");
                break;
            case LinkMonitor::REGISTRATION_STATE_ROAMING:
                detectRegState = false;
                DBG("regState: ROAMING");
                break;
            default:
                DBG("regState: ERROR. Failing.");

        }
      }
  Thread::wait(500);
  int ret = connectionManager.getLinkState(&rssi, &regState, &bearer);
  }

  // let the boss man know I'm alive and entering my main loop  
  ret = connectionManager.sendSM(MY_PHONE_NUMBER, "Hello from Vodafone door controller 2013 - Alarm Active");
  ret = connectionManager.sendSM(BACKUP_NUMBER, "Hello from Vodafone door controller 2013- Alarm Active");

  // better let the boss know that I'm on a roamed network as this could cost some money! or present some routing problems
  if (regState==LinkMonitor::REGISTRATION_STATE_ROAMING)
  {
      //ret = connectionManager.sendSM(MY_PHONE_NUMBER, "Warning sir - I'm on a roamed network!");
      ret = connectionManager.sendSM(BACKUP_NUMBER, "Warning sir - I'm on a roamed network!");
  }
    


  while(true)
  {
    
    // if this state variable has been set I need to keep the doorContactor on until the door is open
    if ((openTheDoor) && (read_digital() == 0))
    {
        DBG("The door is open now and I'm switching off the relay and reseting the state variable");
        doorContactorRelay = 0;                             // door must be open now so switch relay off.
        openTheDoor = false;                                // the door is open now so set it to false.
        led3 = 0;                                           // switch the LED light off to indicate I'm switching the relay off
        closeTheDoor = true;                                // now set the flag to indicate we are waiting for a closed door event allowing us to then activate the alarm
        DBG("The closeTheDoor is now set to TRUE");

    }
    
    if ((read_digital() != 0) && ( closeTheDoor == true))
    {
        closeTheDoor = false;                               // the door is now closed after a legal open so reset the flag
        alarmActive = true;                                 // OK we can now activate the ALARM again
        led4 = 1;                                           // switch the alarm light on
        DBG("The door is now closed after an open and the ALARM is no re-activated");
    
    }
    
    // check to see if the door is open and the alarm is on - if it is send warning message - but just once!!!!
    if ((read_digital() == 0) && (alarmActive == true))
    {
        DBG("The door is now open - but the ALARM is active - send emergency text to warn our boss!");
        DBG("*******************************************************************");
        DBG("WARNING - WARNING - WARNING - WARNING - WARNING - WARNING - WARNING");
        DBG("*******************************************************************");
        alarmActive = false;    // set the alarm off as we don't want to keep sending text messages
        led4 = 0;               // set the alarm light to off
        connectionManager.sendSM(BACKUP_NUMBER, "ALARM - ALARM - ALARM - Your door is open but I've not received an 'open' command!!!" );
        connectionManager.sendSM(MY_PHONE_NUMBER, "ALARM - ALARM - ALARM - Your door is open but I've not received an 'open' command!!!" );
        
    }
    
    // check to see that the door is closed and the alarm can be reset. this will happen if the openTheDoor state variable
    // is FALSE (indicating that the system is not trying to hold the relay open to allow accesss, the door is closed and the ALARM is false
    if ((openTheDoor == false) && (read_digital() !=0) && (alarmActive == false))
    {
        DBG("Resetting the ALARM to active again");
        alarmActive = true;     // set the alarm on as the door is closed again
        led4 = 1;
    }
        
        
    // check the state of the door and switch the LED light to 'ON' for door open and 'OFF' for door closed.
    //if (doorProximitySwitch.read() == 0)
    if (read_digital() == 0)
    {
        // DBG("The door proximity switch is: %d so I think it's open", doorProximitySwitch.read() );
        led1 = 1;
    }
    else
    {
        // DBG("The door proximity switch is: %d so I think it's closed", doorProximitySwitch.read() );
        led1 = 0;
    }
    
    
 
    int ret = connectionManager.getSMCount(&count);         // check to see if there is an incoming SMS message
    if(ret)
    {
      WARN("getSMCount returned %d", ret);
      Thread::wait(125);
      continue;
    }
        
    // DBG("The SMS count is now at: %d for this modem", count);
    if( count > 0)                                          // if there are messages in the mailbox start pulling them off the queue
    {
      DBG("%d SMS to read", count);
      ret = connectionManager.getSM(num, msg, 160);
    
      if(ret)
      {
        WARN("getSM returned %d", ret);
        Thread::wait(125);
        continue;
      }

      DBG("The message is from number: %s and the message is: \"%s\"", num, msg);
      
      // if the SMS is from your own number, ignore it!
      if(strcmp(num,"+447785666088")==0) 
      {
        DBG("The message appears to be from my MSISDN number: %s, I'll do nothing.", num);
        continue;
      }
      
  
      
      if ((strcmp (msg, "open") ==0)|| (strcmp(num,"TrackaPhone")==0)) 
      {
        DBG("The SMS message indicates I should open the door");
        
        // check the door is not already open if it is do nothing and send a warning back to the sender
        // for this door the switch 'true' then its closed - we have to have an active signal to show it's securely closed

        if (read_digital() == 0 )
        {
            DBG("The door is already open - so I'll do nothing");
            connectionManager.sendSM(num, " WARNING - The door is already open sir! ");
        }
        else
        {
            doorContactorRelay = 1;
            openTheDoor = true;     // set the state vairable to let the door contoller know it needs to open the door
            alarmActive = false;    // set the alarm OFF until the door is open
            led4 = 0;               // set the alarm light to off
            DBG("The ALARM is now set to %d: ", alarmActive);
            led3 = 1;
            DBG("The relay has been activated to open the door");
            connectionManager.sendSM(num, " Door open sir... Welcome home! " );
        }
      } 
      else 
      {
        DBG("The SMS message is not recognized");
        connectionManager.sendSM(num, " Wrong password sir... Try 'open' and I'll open! ;-) ");
      }
      
        
      
    }
    Thread::wait(500);
    DBG("Main loop. Make sure we kick the dog to stop a restart happening!");
    watchDogTimer.kick();
  }
}

void keepAlive(void const*) {
   while(1)
   {
     led2=!led2;
     Thread::wait(500);
   }
}


int main() {
    // On reset, indicate a watchdog reset has happened by switching all lights on for 3 seconds
    if ((LPC_WDT->WDMOD >> 2) & 1)
    {
        led1 = 1;
        led2 = 1;
        led3 = 1;
        led4 = 1;
        Thread::wait(3000);
        led1 = 0;
        led2 = 0;
        led3 = 0;
        led4 = 0;

    }

        
    // setup a 30 second timeout on watchdog timer hardware
    // needs to be longer than worst case main loop exection time
    watchDogTimer.setTimeOut(30.0);  
 
    DBG_INIT();
    DBG_SET_SPEED(115200);
    doorController(NULL);
    //Thread testTask(test, NULL, osPriorityNormal, 1024*5);

    keepAlive(NULL);

    
  return 0;
}
