#include "mbed.h"
#include "cmd_line.h"
#include <vector>
#include <string>

#define SHELL_BUFFER_LENGHT 64

static int32_t shell_reset(uint8_t* argv);
static int32_t shell_help(uint8_t* argv);

cmd_line_t lgn_cmd_table[] = {

    /*************************************************************************/
    /* system command */
    /*************************************************************************/
    {(const int8_t*)"help",     shell_help,         (const int8_t*)"help info"},
    {(const int8_t*)"reset",    shell_reset,        (const int8_t*)"reset terminal"},


    /* End Of Table */
    {(const int8_t*)0,(pf_cmd_func)0,(const int8_t*)0}
};

Serial console(D1, D0, 115200);

vector<string> command_vector;

struct shell_t {
    uint8_t index;
    uint8_t data[SHELL_BUFFER_LENGHT];
};

volatile struct shell_t shell;

void console_rx_callback()
{
    volatile uint8_t c = console.getc();
    if (shell.index < SHELL_BUFFER_LENGHT - 1) {

        if (c == '\r' || c == '\n') { /* linefeed */

            console.putc('\r');
            console.putc('\n');

            shell.data[shell.index] = c;
            shell.data[shell.index + 1] = 0;

            string str_cmd = (const char*)shell.data;

            command_vector.push_back(str_cmd);

            shell.index = 0;
        } else {

            console.putc(c);

            if (c == 8 && shell.index) { /* backspace */
                shell.index--;
            } else {
                shell.data[shell.index++] = c;
            }
        }
    } else {
        console.printf("\nerror: cmd too long, cmd size: %d, try again !\r\n", SHELL_BUFFER_LENGHT);
        shell.index = 0;
    }
}

static void task_shell();

int main()
{
    console.printf("Started Application !\r\n");
    console.attach(console_rx_callback, Serial::RxIrq);
    while (1) {
        task_shell();
    }
}

void task_shell()
{
    if (!command_vector.empty()) {
        string console_command = command_vector.back();
        uint8_t* c_str = (uint8_t*)console_command.c_str();
        uint8_t fist_char = c_str[0];

        switch (cmd_line_parser(lgn_cmd_table, (uint8_t*)&shell.data[0])) {
            case CMD_SUCCESS:
                break;

            case CMD_NOT_FOUND: {
                if (fist_char != '\r' &&
                        fist_char != '\n') {
                    console.printf("cmd unknown\r\n");
                }
            }
            break;

            case CMD_TOO_LONG: {
                console.printf("cmd too long\r\n");
            }
            break;

            case CMD_TBL_NOT_FOUND: {
                console.printf("cmd table not found\r\n");
            }
            break;

            default: {
                console.printf("cmd error\r\n");
            }
            break;
        }

        console.printf("#");

        command_vector.pop_back();
    }
}

int32_t shell_reset(uint8_t* argv)
{
    console.printf("shell_reset: %s\r\n", argv);
    return 0;
}


int32_t shell_help(uint8_t* argv)
{
    uint32_t idx = 0;
    switch (*(argv + 4)) {
        default:
            console.printf("\nCOMMANDS INFORMATION:\r\n\n");
            while(lgn_cmd_table[idx].cmd != (const int8_t*)0) {
                console.printf("%s\n\t%s\r\n\n", lgn_cmd_table[idx].cmd, lgn_cmd_table[idx].info);
                idx++;
            }
            break;
    }
    return 0;
}
