/*
    RadarDemo.h - Simple radar simulator demo, example of graphic commands usage to create 2D graphics.

    Copyright(c) 2016 karpent at gmail.com, MIT License

    Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files(the "Software"),
    to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and / or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions :

    The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
    OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
    THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

/**
  * Usage example:
  *
@verbatim
    #include "RK043FN48H.h"
    #include "RadarDemo.h"

    RK043FN48H display;

    int main()
    {
        RadarDemo demo(&display);
        demo.Initialize();

        while(demo.IsRunning()) {
            demo.Render();
            wait(0.04f);
        }
    }
@endverbatim
*/

#pragma once

#include "Radar.h"
#include "RadarTimer.h"
#include "TouchScreen.h"

typedef enum EUserInputType
{
    None = 0,
    SingleTouch =1,
    MultiplyTouch = 2,
    ButtonPressed = 3
} UserInputType;


/// <summary>
/// Radar display demo class.
/// </summary>
/// <seealso cref="Radar" />
class RadarDemo : public Radar
{
public:

    /// <summary>
    /// Duration time of the demo in milisecons.
    /// Set <c>demoTime</c> to 0 if you do not need the time limit.
    /// Set for 0
    /// </summary>
    uint32_t demoTime;

    /// <summary>
    /// Class constructor.
    /// </summary>
    /// <param name="display">The display.</param>
    RadarDemo(Display* display);

    /// <summary>
    /// Class destructor.
    /// </summary>
    ~RadarDemo();

    /// <summary>
    /// Initializes radar demo instance. Generates sample data for the demo.
    /// </summary>
    void Initialize();

    /// <summary>
    /// Renders single frame of radar imaging.
    /// </summary>
    void Render();

    /// <summary>
    /// Determines whether this instance is running.
    /// </summary>
    /// <returns>
    ///   <c>true</c> if this instance is running; otherwise, <c>false</c>.
    /// </returns>
    bool IsRunning();

    /// <summary>
    /// Check if background needs redraw while range or center location changes.
    /// </summary>
    bool NeedsRedraw();

    /// <summary>
    /// @brief Force background redraw.
    /// Sets the display state for NEEDS REDRAW.
    /// </summary>
    void UnvalidateBackground();

    /// <summary>
    /// Generate sample targets.
    /// </summary>
    /// <param name="count">The number of targets to create.</param>
    void AddSampleTargets(int count);

    /// <summary>
    /// Detect a touch type on the screen in given area.
    /// </summary>
    /// <param name="screen">The screen window coordinates.</param>
    UserInputType CheckUserInput(Window screen);
    
    RadarTimer timer;

private:

    float currentBeamAngle;
    float lastBeamAngle;

    bool _needsRedraw;
    
    TouchScreen* _ts;
    uint32_t _buttonPressTime;
    bool TouchScreenDisabled();  
    void SetTouchScreenTimeout();  
    
};

