//
// RadarDemo.cpp - example of graphic commands usage to create 2D graphics.
//

#include "RadarDemo.h"
#include "RK043FN48H.h"

// Size of range control areas on the screen
const int raSize = 50;

// Button Press Timeout in miliseconds
const uint32_t buttonPressTimeout = 400;


RadarDemo::RadarDemo(Display* display) : Radar(display)
{
    _ts = new TouchScreen(display);
}


RadarDemo::~RadarDemo()
{
    Radar::~Radar();
    free(_ts);
}


void RadarDemo::Initialize()
{
    // Set scan perion for 6 seconds
    timer.SetScanPeriod(6);

    // Run forever, to set limited time for the demo
    // use for example: demoTime = 10 * scanPeriod;
    demoTime = 0;

    // Put a number of sample targets on the display
    AddSampleTargets(30);

    // Use medium range, values 0..2 are available now
    SetRange(1);

    // Remark : Data member initializer is not allowed for ARM compiler,
    //          initialize data in class constructor.
    currentBeamAngle = 0;
    lastBeamAngle = currentBeamAngle;

    // Force background redraw
    UnvalidateBackground();
    
    _buttonPressTime = 0;

    // Initialize touchscreen
    _ts->Init();
   
    timer.Start();   
}


void RadarDemo::Render()
{
    timer.RegisterScan();

    // Calculate actual beam position
    lastBeamAngle = currentBeamAngle;
    currentBeamAngle = timer.GetBeamAngle();

    CheckUserInput(window); 

    RK043FN48H* display = (RK043FN48H*)GetDisplay();
    if(NeedsRedraw()) {

        display->SetActiveLayer(Background);
        display->Clear();

        // Set draw color before calling draw method
        display->SetDrawColor(0x80, 0, 0, 0xFF);
        DrawRangeButtons(raSize);

        display->SetDrawColor(0x40, 0x40, 0x40, 0xFF);
        DrawMarkers();

        display->SetDrawColor(0x80, 0x80, 0x80, 0xFF);
        DrawBorder();

        display->SetActiveLayer(Foreground);
        _needsRedraw = false;
    }

    UpdateTargetsLocation(lastBeamAngle, currentBeamAngle, timer.GetRunningTime());

    // Redraw foreground
    display->Clear();
    DrawTracks();

    DrawRadarBeam(currentBeamAngle);
}


bool RadarDemo::IsRunning()
{
    return demoTime > 0 ? timer.GetRunningTime() <= demoTime : true;
}


void RadarDemo::AddSampleTargets(int count)
{
    const float minSpeed = 200; // [km/h]
    const float maxSpeed = 800; // [km/h]
    Target *target;

    srand(timer.GetRunningTime());

    for (int i = 0; i<count; i++) {
        // Generate target parameters
        float angle = 2 * M_PI * rand() / (float)RAND_MAX;
        float range = GetMaxRange() * rand() / (float)RAND_MAX;
        float speed = minSpeed + (maxSpeed - minSpeed) * rand() / (float)RAND_MAX;
        float direction = 2 * M_PI * rand() / (float)RAND_MAX;

        // Create a new target
        target = new Target(i + 100, speed, direction);
        target->SetLocationAngular(range, angle);

        // Put target on the list
        targetsList.push_back(*target);
    }
}


void RadarDemo::UnvalidateBackground()
{
    _needsRedraw = true;
}


bool RadarDemo::NeedsRedraw()
{
    return _needsRedraw;
}


UserInputType RadarDemo::CheckUserInput(Window screen)
{
    pPoint p;
    
    // There is a timeout after pressing the button
    if(TouchScreenDisabled())
        return None;
        
    if ((p = _ts->DetectTouch()) != NULL) {

        if(p->X > (screen.x2-raSize) && p->Y < (screen.y1 + raSize)) {
            if(ChangeRange(1)) {
                UnvalidateBackground();
                SetTouchScreenTimeout();
                
                return ButtonPressed;
            }
        }

        if(p->X > (screen.x2-raSize) && p->Y > (screen.y2-raSize)) {
            if(ChangeRange(-1)) {
                UnvalidateBackground();
                SetTouchScreenTimeout();
                
                return ButtonPressed;
            }
        }

        SetCenter(p->X, p->Y);
        UnvalidateBackground();
        
        return SingleTouch;
    }

    return None;
}

bool RadarDemo::TouchScreenDisabled()
{
    return _buttonPressTime + buttonPressTimeout > timer.GetRunningTime();
}

void RadarDemo::SetTouchScreenTimeout()
{
    _buttonPressTime = timer.GetRunningTime();
}
