/*
    Target.h - Simple simulation of radar target.

    Copyright(c) 2016 karpent at gmail.com, MIT License

    Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files(the "Software"),
    to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and / or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions :

    The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
    OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
    THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/ 

#pragma once

#include "Location.h"
#include "mbed.h"   // for uint32_t

/**
  * @brief Simple simulation of radar target.
  */
class Target
{
public:
    /// <summary>
    /// Initializes a new instance of the <see cref="Target"/> class.
    /// </summary>
    /// <param name="id">The identifier.</param>
    /// <param name="speed">The speed.</param>
    /// <param name="direction">The direction.</param>
    Target(int id, float speed, float direction);

    Target(int id, float x, float y, float h, float speed, float direction);

    /// <summary>
    /// Sets the location angular.
    /// </summary>
    /// <param name="distance">The distance.</param>
    /// <param name="azimuth">The azimuth.</param>
    void SetLocationAngular(float distance, float azimuth);

    /// <summary>
    /// Gets the location of target.
    /// </summary>
    /// <returns>Target location</returns>
    Location GetLocation();
    
    /// <summary>
    /// Sets target location.
    /// </summary>
    /// <param name="x">x value.</param>
    /// <param name="y">y value.</param>
    void SetLocation(float x, float y);

    /// <summary>
    /// Update target location for the specified time in miliseconds.
    /// </summary>
    /// <param name="currentTime">The current time.</param>
    void UpdateLocationForTime(uint32_t currentTime);

    float GetX();
    float GetY();
    float GetAzimuth();
    float GetDistance();
    
    float GetSpeed();
    float GetDirection();
    int GetType();

    int Id;

protected:
    Location _location;
    float _speed;
    float _direction;
    int _type;
    uint32_t _lastUpdateTime;
};

