/*
    Canvas.h - Simple canvas class declaration

    Copyright(c) 2016 karpent at gmail.com, MIT License

    Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files(the "Software"),
    to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and / or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions :

    The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
    OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
    THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#pragma once

#include "Commons.h"
#include "Display.h"

#define PLANE_BITMAP_ELEMENT_BITS 8

/**
*  @brief Canvas is a frame buffer where everything is drawn,
*  uses its own graphics commands to create graphics.
*  Cavas bitmap must be copied to the display to make image visible.
*/
class Canvas : public Display
{
public:

    /// <summary>
    /// Initializes a new instance of the <see cref="Canvas"/> class.
    /// </summary>
    /// Class constructor
    Canvas();

    /// <summary>
    /// Initializes a new instance of the <see cref="Canvas"/> class.
    /// </summary>
    /// <param name="width">The width.</param>
    /// <param name="height">The height.</param>
    Canvas(uint16_t width, uint16_t height);

    /// <summary>
    /// Finalizes an instance of the <see cref="Canvas"/> class.
    /// </summary>
    ~Canvas();

    /// <summary>
    /// Clears the canvas.
    /// </summary>
    void Clear();

    /// <summary>
    /// Gets the bitmap.
    /// </summary>
    /// <returns></returns>
    uint8_t* GetBitmap();

    /// <summary>
    /// Returns screen width.
    /// </summary>
    /// <returns></returns>
    uint16_t virtual DisplayWidth();

    /// <summary>
    /// Redurns screen height.
    /// </summary>
    /// <returns></returns>
    uint16_t virtual DisplayHeight();

    /// <summary>
    /// Draws the point.
    /// </summary>
    /// <param name="posX">The position x.</param>
    /// <param name="posY">The position y.</param>
    /// <param name="colorMask">The color mask.</param>
    void virtual DrawPoint(int posX, int posY, uint32_t colorMask);

    /// <summary>
    /// Sets the size.
    /// </summary>
    /// <param name="width">The width.</param>
    /// <param name="height">The height.</param>
    /// <returns></returns>
    bool SetSize(uint16_t width, uint16_t height);

    /// <summary>
    /// Determines whether this instance is set.
    /// </summary>
    /// <returns>
    ///   <c>true</c> if this instance is set; otherwise, <c>false</c>.
    /// </returns>
    bool IsSet();

private:
    /// <summary>
    /// Resets this instance.
    /// </summary>
    void Reset();

    uint16_t _width;
    uint32_t _height;

    /*
    *  Canvas plane bitmap - depth 1.
    */
    uint8_t *_planeBitmap;
    uint32_t _planeBitmapSize;
};
