//
// RK043FN48H.h - Graphics commands for DISCO_F746NG display build on top of stm32746g driver.
//

#pragma once

#include "Commons.h"
#include "Display.h"
#include "../BSP_DISCO_F746NG/Drivers/BSP/STM32746G-Discovery/stm32746g_discovery_lcd.h"

#define ALPHA_MASK 0x00FFFFFF

#define TITLE "Signal Process"
#define XPOS_TITLE 10
#define YPOS_TITLE 10
/**
 * @brief Display layers available for RK043FN48H
 */
typedef enum ELayer {
    Background = 0, // Bottom
    Foreground = 1  // Upper
} Layer;

/**
  * @brief DISCO_F746NG display
  */
class RK043FN48H : public Display
{
public:
    RK043FN48H();
    ~RK043FN48H();

    /**
    * @brief  Clear the active layer using background color.
    * @retval None
    */
    void Clear();

    /**
    * @brief  Clear the active layer using given color.
    * @retval None
    */
    void Clear(uint32_t color);

    /**
    * @brief Clear selected layer using given color.
    * @retval None
    */
    void ClearLayer(Layer layer, uint32_t color);
    
    /**
    * @brief  Set background color for active layer.
    * @retval None
    */
    void SetBackgroundColor(uint32_t color);

    /**
    * @brief  Set foreground color for active layer.
    * @retval None
    */
    void SetForegroundColor(uint32_t color);
    
    /// <summary>
    /// Sets the color of the draw.
    /// </summary>
    /// <param name="red">The red.</param>
    /// <param name="green">The green.</param>
    /// <param name="blue">The blue.</param>
    /// <param name="alpha">The alpha.</param>
    void virtual SetDrawColor(uint8_t red, uint8_t green, uint8_t blue, uint8_t alpha);
    
    /// <summary>
    /// Returns the color to draw on selected layer.
    /// </summary>
    uint32_t virtual GetDrawColor();
    
    /// <summary>
    /// Draws the point.
    /// </summary>
    /// <param name="posX">The position x.</param>
    /// <param name="posY">The position y.</param>
    /// <param name="colorMask">The color mask.</param>
    void virtual DrawPoint(int posX, int posY, uint32_t colorMask);
      
    //void virtual DrawLine(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2);

    //void virtual FillRectangle(uint16_t posX, uint16_t posY, uint16_t x2, uint16_t y2);

    void CopyBitmap(Layer layer, uint8_t *bitmap, uint32_t width, uint32_t height, uint32_t rgbGolorCode);

    /// <summary>
    /// Returns screen width.
    /// </summary>
    /// <returns></returns>
    uint16_t virtual DisplayWidth();

    /// <summary>
    /// Returns screen height.
    /// </summary>
    /// <returns></returns>
    uint16_t virtual DisplayHeight();
    
    void SetActiveLayer(Layer layer);
    
    Layer GetActiveLayer();
    
    void SetLayersTransparency( uint8_t background, uint8_t foreground);
    
    void SetLayersVisibility( bool background, bool foreground);
    

    //New Function
    //Add date: 25/08/2019


    /**
    * @brief  Draws an horizontal line.
    * @param  Xpos: X position
    * @param  Ypos: Y position
    * @param  Length: Line length
    * @retval None
    */
  void DrawHLine(uint16_t Xpos, uint16_t Ypos, uint16_t Length);

  /**
    * @brief  Draws a vertical line.
    * @param  Xpos: X position
    * @param  Ypos: Y position
    * @param  Length: Line length
    * @retval None
    */
  void DrawVLine(uint16_t Xpos, uint16_t Ypos, uint16_t Length);

  /**
    * @brief  Draws an uni-line (between two points);.
    * @param  x1: Point 1 X position
    * @param  y1: Point 1 Y position
    * @param  x2: Point 2 X position
    * @param  y2: Point 2 Y position
    * @retval None
    */
  void DrawLine(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2);

  /**
    * @brief  Draws a rectangle.
    * @param  Xpos: X position
    * @param  Ypos: Y position
    * @param  Width: Rectangle width  
    * @param  Height: Rectangle height
    * @retval None
    */
  void DrawRect(uint16_t Xpos, uint16_t Ypos, uint16_t Width, uint16_t Height);

  /**
    * @brief  Draws a circle.
    * @param  Xpos: X position
    * @param  Ypos: Y position
    * @param  Radius: Circle radius
    * @retval None
    */
  void DrawCircle(uint16_t Xpos, uint16_t Ypos, uint16_t Radius);

  /**
    * @brief  Draws an poly-line (between many points);.
    * @param  Points: Pointer to the points array
    * @param  PointCount: Number of points
    * @retval None
    */
  void DrawPolygon(pPoint Points, uint16_t PointCount);

  /**
    * @brief  Draws an ellipse on LCD.
    * @param  Xpos: X position
    * @param  Ypos: Y position
    * @param  XRadius: Ellipse X radius
    * @param  YRadius: Ellipse Y radius
    * @retval None
    */
  void DrawEllipse(int Xpos, int Ypos, int XRadius, int YRadius);

  /**
    * @brief  Draws a pixel on LCD.
    * @param  Xpos: X position
    * @param  Ypos: Y position
    * @param  RGB_Code: Pixel color in ARGB mode (8-8-8-8);
    * @retval None
    */
  void DrawPixel(uint16_t Xpos, uint16_t Ypos, uint32_t RGB_Code);

  /**
    * @brief  Draws a bitmap picture loaded in the internal Flash in ARGB888 format (32 bits per pixel);.
    * @param  Xpos: Bmp X position in the LCD
    * @param  Ypos: Bmp Y position in the LCD
    * @param  pbmp: Pointer to Bmp picture address in the internal Flash
    * @retval None
    */
  void DrawBitmap(uint32_t Xpos, uint32_t Ypos, uint8_t *pbmp);


    /**
    * @brief  Draws a full rectangle.
    * @param  Xpos: X position
    * @param  Ypos: Y position
    * @param  Width: Rectangle width  
    * @param  Height: Rectangle height
    * @retval None
    */
    void FillRect(uint16_t Xpos, uint16_t Ypos, uint16_t Width, uint16_t Height);
    /**
    * @brief  Draws a full circle.
    * @param  Xpos: X position
    * @param  Ypos: Y position
    * @param  Radius: Circle radius
    * @retval None
    */
    void FillCircle(uint16_t Xpos, uint16_t Ypos, uint16_t Radius);

    /**
    * @brief  Draws a full poly-line (between many points);.
    * @param  Points: Pointer to the points array
    * @param  PointCount: Number of points
    * @retval None
    */
    void FillPolygon(pPoint Points, uint16_t PointCount);

    /**
    * @brief  Draws a full ellipse.
    * @param  Xpos: X position
    * @param  Ypos: Y position
    * @param  XRadius: Ellipse X radius
    * @param  YRadius: Ellipse Y radius  
    * @retval None
    */
    void FillEllipse(int Xpos, int Ypos, int XRadius, int YRadius);


    //Add LCD String Display
    /**
    * @brief  Sets the LCD text color.
    * @param  Color: Text color code ARGB(8-8-8-8);
    * @retval None
    */
  void SetTextColor(uint32_t Color);

  /**
    * @brief  Gets the LCD text color.
    * @retval Used text color.
    */
  uint32_t GetTextColor(void);

  /**
    * @brief  Sets the LCD background color.
    * @param  Color: Layer background color code ARGB(8-8-8-8);
    * @retval None
    */
  void SetBackColor(uint32_t Color);

  /**
    * @brief  Gets the LCD background color.
    * @retval Used background colour
    */
  uint32_t GetBackColor(void);
  
    /**
    * @brief  Sets the LCD text font.
    * @param  fonts: Layer font to be used
    * @retval None
    */
  void SetFont(sFONT *fonts);

  /**
    * @brief  Gets the LCD text font.
    * @retval Used layer font
    */
  sFONT *GetFont(void);

  /**
    * @brief  Reads an LCD pixel.
    * @param  Xpos: X position 
    * @param  Ypos: Y position 
    * @retval RGB pixel color
    */
  uint32_t ReadPixel(uint16_t Xpos, uint16_t Ypos);

  /**
    * @brief  Clears the selected line.
    * @param  Line: Line to be cleared
    * @retval None
    */
  void ClearStringLine(uint32_t Line);

  /**
    * @brief  Displays one character.
    * @param  Xpos: Start column address
    * @param  Ypos: Line where to display the character shape.
    * @param  Ascii: Character ascii code
    *           This parameter must be a number between Min_Data = 0x20 and Max_Data = 0x7E 
    * @retval None
    */
  void DisplayChar(uint16_t Xpos, uint16_t Ypos, uint8_t Ascii);

  /**
    * @brief  Displays characters on the LCD.
    * @param  Xpos: X position (in pixel);
    * @param  Ypos: Y position (in pixel);   
    * @param  Text: Pointer to string to display on LCD
    * @param  Mode: Display mode
    *          This parameter can be one of the following values:
    *            @arg  CENTER_MODE
    *            @arg  RIGHT_MODE
    *            @arg  LEFT_MODE   
    * @retval None
    */
  void DisplayStringAt(uint16_t Xpos, uint16_t Ypos, uint8_t *Text, Text_AlignModeTypdef Mode);

  /**
    * @brief  Displays a maximum of 60 characters on the LCD.
    * @param  Line: Line where to display the character shape
    * @param  ptr: Pointer to string to display on LCD
    * @retval None
    */
  void DisplayStringAtLine(uint16_t Line, uint8_t *ptr);
private:

    // Frame buffer adresses for layers
    uint32_t FbForegroundStartAdress;
    uint32_t FbBackgroundStartAdress;

    /**
    *   @brief Mirror structure for drivers DropProp,
    *   Contains selected font, background and foreground color
    */
    LCD_DrawPropTypeDef actualDrawProp[MAX_LAYER_NUMBER];
    
};

