#include "mbed.h"
#include "EthernetNetIf.h"
#include "HTTPServer.h"
#include "myconfig.h"
#define MBED
#include "ft.h"
#include "ftlib.h"
#include "ftserver.h"
#include "msgSerial.h"

/****************************************************
 * pin configuration for ft test board
 *
 * 5 mosi for external SD card pin3 pin2=3.3V pin1=GND
 * 6 miso for external SD card pin4
 * 7 sck  for external SD card pin5
 * 11 cs  for external SD card pin6
 *
 * 8 dir for RS485
 * 9  TX for RS485/TTL/RS232(D)  (COM1)
 * 10 RX for RS485/TTL/RS232(D)
 *
 * 13 TX for RS232/MIDI          (COM2)
 * 14 RX for RS232/MIDI
 *
 * 12 spare6
 * 15 spare5
 * 16 spare4
 * 17 spare3
 * 18 spare2 (spare1 is GND)
 *
 * 19 analog1 (analog3 is +3.3V)
 * 20 analog2 (analog4 is GND)
 *
 * 21 motor en3-4
 * 22 motor 3
 * 23 motor 4
 * 24 motor 2
 * 25 motor 1
 * 26 motor en1-2
 *
 * 27 SCL
 * 28 SDA
 *
 * 29 yellow LED on MAGJACK
 * 30 green LED on MAGJACK
*****************************************************/
//#define FIO1PIN (*(unsigned long*)0x2009C034UL)
#define FIO1PIN (LPC_GPIO1->FIOPIN)

#define HWID    1
#define SUBID   100

EthernetNetIf eth;
HTTPServer svr;
ftServer *server;
myConfig config;

LocalFileSystem local("local");
LocalFileSystem fs("webfs");

DigitalOut Green(p30), Yellow(p29);

msgSerial *serialport=0;

FT_TRANSFER_AREA *ta = 0;
FT_HANDLE h = 0;
int trigger_interface = 0;

char *names[] = {"NO_FT_DEVICE","FT_AUTO_TYPE","FT_INTELLIGENT_IF","FT_INTELLIGENT_IF_SLAVE","FT_ROBO_IF_IIM",
                 "FT_ROBO_IF_USB","FT_ROBO_IF_COM","FT_ROBO_IF_OVER_RF","FT_ROBO_IO_EXTENSION","FT_ROBO_RF_DATA_LINK"
                };
int types[] = {NO_FT_DEVICE,FT_AUTO_TYPE,FT_INTELLIGENT_IF,FT_INTELLIGENT_IF_SLAVE,FT_ROBO_IF_IIM,
               FT_ROBO_IF_USB,FT_ROBO_IF_COM,FT_ROBO_IF_OVER_RF,FT_ROBO_IO_EXTENSION,FT_ROBO_RF_DATA_LINK,NO_FT_DEVICE//this last one is the default
              };
char *modes[] = {"cable","ftlib","serial"};
int type = NO_FT_DEVICE;
int mode = 0;

unsigned short encode(char*);
void callback(void*ctx);
void *context = 0;
void messageCB(SMESSAGE*);
NOTIFICATION_EVENTS ne = {callback, context, 0, 0, 0, messageCB};
unsigned short msgID = encode("set");

void http_server() {
    FSHandler::mount("/webfs", "/files"); //Mount /webfs path on /files web path
    svr.addHandler<FSHandler>("/files");
    svr.bind(80);
    printf("Listening for http requests...\n");
}

unsigned short encode(char* s) { //convert 3 character command to 16bit value
    unsigned short r = 0;
    for (int i = 0; i < 3 && *s != '\0'; i++) {
        r *= 40;
        if (s[i] >= '0' && s[i] <= '9') {
            r += s[i];
            r -= '0' - 27;
        } else if (s[i] >= 'A' && s[i] <= 'Z') {
            r += s[i];
            r -= 'A' - 1;
        } else if (s[i] >= 'a' && s[i] <= 'z') {
            r += s[i];
            r -= 'a' - 1;
        } else
            switch (s[i]) {
                case '+':
                    r += 37;
                    break;
                case '-':
                    r += 38;
                    break;
                case '!':
                    r += 39;
                    break;
            }
    }
    return r;
}

void handleMessage(char m[7]) { //called when a valid tcpip message is received
    SMESSAGE *msg = (SMESSAGE*)(m+1);
    switch (mode) {
        case 0://cable
            if (msg->ucHwId == HWID && msg->ucSubId == SUBID && msg->uiMsgId == msgID && ta != 0) { //check message values
                ta->M_Main = msg->ucB2; //long cable semantics, copy the message value to the motor outputs (16 bit only!)
                ta->M_Sub1 = msg->ucB3;
            }
            break;
        case 1://ftlib, not (yet) implemented
            if (h)
                SendFtMessage(h, msg->ucHwId, msg->ucSubId, msg->dw, 0, MSG_SEND_NORMAL); //use the ftlib function to forward the message to the interface
            break;
        case 2://serial
            if (serialport)
                serialport->write(m);
            break;
    }
}

void OnSerMsg(char *m) {
    if (server && mode==2)//serial
        server->writeStream((SMESSAGE*)(m+1));
}

void callback(void*ctx) { //new data from interface has arrived
    if (mode==0 && ta->ChangeEg) {//cable
        SMESSAGE m = { {HWID, SUBID} };
        m.uiMsgId = msgID;
        m.ucB2 = ta->E_Main;
        m.ucB3 = ta->E_Sub1;
        if (server)
            server->writeStream(&m);
//        printf("%02X\n", ta->E_Main);
    }
}

void messageCB(SMESSAGE *msg) {// not (yet) implemented
    if (server && mode==1)//ftlib
        server->writeStream(msg);
}

void init_ft_interface() {
    char comport[5]= "COM1";
    InitFtLib();
    unsigned ver = GetLibVersion();
    printf("libver = %08X\n", ver);

    if (!config.read("/local/ft.ini"))
        printf("reading of configfile failed!\n");
    mode = config.getLookup("mode", modes, 3);
    switch (mode) {
        case 0://cable (requires ftlib too)
        case 1://ftlib
            type =  types[config.getLookup("FtInterface", names, sizeof(names)/sizeof(char*))];
            if (type==FT_INTELLIGENT_IF || type==FT_INTELLIGENT_IF_SLAVE || type==FT_ROBO_IF_IIM || type==FT_ROBO_IF_COM) {//serial device
                if (!config.getValue("FtCommPort", comport, sizeof(comport)))
                    printf("FtCommPort not found, using default\n");
                printf("opening port %s by Ftlib\n", comport);
                h = OpenFtCommDevice(comport, type, 5);
                if (h) {
                    printf("OpenFtCommDevice on %s succeeded\n", comport);
                    //long v = GetFtFirmware(h);
                    //printf("FW: %08lX\n", v);
                    ta = GetFtTransferAreaAddress(h);//always get before starting otherwise not imm avail in callback
                    int err = StartFtTransferAreaWithCommunication(h, &ne);
                    if (err==0) {
                        printf("transfer started\n");
                        for (int i = 0; i < 8; i++) {
                            ta->MPWM_Main[i] = 7;
                            ta->MPWM_Sub1[i] = 7;
                        }
                    } else
                        printf("Error %08X: %s\n", err, GetFtLibErrorString(err,1));
                }
                printf("Opening Ft device failed\n");
            } else {
                printf("devicetype %d is not supported\n", type);
            }
            break;
        case 2://serial
            if (!config.getValue("FtCommPort", comport, sizeof(comport)))
                printf("FtCommPort not found, using default\n");
            printf("opening port %s for messaging\n", comport);
            switch (comport[3]) {
                case '1':
                    serialport = new msgSerial(p9, p10, OnSerMsg);
                    break;
                case '2':
                    serialport = new msgSerial(p13, p14, OnSerMsg);
                    break;
                case '3':
                    serialport = new msgSerial(p28, p27, OnSerMsg);
                    break;
                default:
                    // serialport = &viaUsb;
                    printf("USB testoutput not supported for messages\n");
                    break;
            }
            break;
        default:
            printf("Mode %d is not supported or unknown\n",mode);
    }
}

int main() {
    init_ft_interface();

    printf("Setting up ethernet...\n");
    EthernetErr ethErr = eth.setup();
    if (ethErr) {
        printf("Error %d in ethernet setup.\n", ethErr);
        return -1;
    }
    printf("Setup of ethernet OK\n");

    printf("starting http server\n");
    http_server();

    printf("creating ft socket server\n");
    server = new ftServer(IpAddr(),config.getInt("port",5000));

    printf("starting ft socket server\n");
    if (server->startServer()) {
        for (;;) {//main loop
            Net::poll();
//            server->pollServer();
            if (trigger_interface) {
                if (h->guardedFtThreadBegin())
                    trigger_interface = 0;
            }
            Green = !(FIO1PIN&(1<<25));//link
            Yellow= !(FIO1PIN&(1<<26));//speed
        }
    } else
        printf("server could not be started\n");

    if (server)
        delete server;
    if (serialport)
        delete serialport;
    if (h) {
        StopFtTransferArea(h);
        CloseFtDevice(h);
    }
    CloseFtLib();
    printf("end of program");
    return 0;
}
