/* mbed TextLCD23017 Library, for a 4-bit LCD based on HD44780
 * Copyright (c) 2007-2010, sford, http://mbed.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include <stdarg.h>
#include "TextLCD23017.h"
#include "mbed.h"

/**************************************************************
Port A: databus 8-bit bidir, also used as input for keys (A2-A7)
IntA wired to mbed pin??
PortB: B0 out: keys out, set low to read keys on PortA
       B1 in:  rot.enc. push button act.L
       B2 out: buzzer
       B3 in:  rot.enc. A
       B4 in:  rot.enc. B
       B5 out: LCD E (CS\)
       B6 out: LCD R\W (WR\)
       B7 out: LCD RS (A0)
**************************************************************/
TextLCD23017::TextLCD23017(MCP23017& intf, LCDType type) : _intf(intf), _type(type) {
    ::printf("creating lcd\n");
    wait(0.015);        // Wait 15ms to ensure powered up of LCD
//init the MCP23017
    intf.direction(PORT_A, PORT_DIR_IN); //input
    intf.direction(PORT_B, INPUTS);// rot enc input
    intf.configurePullUps(PORT_A, PORT_DIR_IN);
    intf.configurePullUps(PORT_B, INPUTS);
    intf.write(PORT_B, 0x00); //enable keys
    //intf.interruptPolarity(ACTIVE_LOW);
    intf.mirrorInterrupts(true);
    
//initialisation according to HD44780 datasheet
    // send "Display Settings" 2 times for 8 bit interface
    writeByte(0x30);
    wait(0.0041);  // this command takes 4.1ms, so wait for it
    writeByte(0x30);


    writeByte(0x38);     // 8-bit mode, 2 lines 8x5 font
    //wait(0.000040f);    // most instructions take 40us
    writeByte(0x08);  //display off, cursor off
    writeByte(0x01);  //clear display
    writeByte(0x06);  //increment, no shift
//from here on busy can be tested

    writeCommand(0x0F); //display on, cursor on, blink on
//    cls();
::printf("lcd created\n");
}

int TextLCD23017::printf (char * format, ...) {
    char buffer[40];
    va_list args;
    va_start (args, format);
    int rv=vsprintf (buffer,format, args);
//    ::printf("printing:'%s'\n", buffer);
    writeString (buffer);
    va_end (args);
    return rv;
}

//busy not used, i2c is slow enough
bool TextLCD23017::busy() {//assume PORT_A is input, PORT_A intr disabled, takes minimum 10bytes ~1ms
//setup and hold of RS-E are not met!
    _intf.write(PORT_B, RW|E|KEYS); //read R=1, E=1, disable keys
    _ar = _intf.read(PORT_A);
    _intf.write(PORT_B, KEYS); //read R=0, E=0
    return (_ar & 0x80) ;//return with keys disabled
}

void TextLCD23017::character(int column, int row, int c) {
    int a = address(column, row);
    writeCommand(a);
    writeData(c);
}

void TextLCD23017::cls() {
    writeCommand(0x01); // cls, and set cursor to 0
    wait(0.00164f);     // This command takes 1.64 ms
    locate(0, 0);
}

void TextLCD23017::locate(int column, int row) {
    _column = column;
    _row = row;
}

int TextLCD23017::_putc(int value) {
    if (value == '\n') {
        _column = 0;
        _row++;
        if (_row >= rows()) {
            _row = 0;
        }
    } else {
        character(_column, _row, value);
        _column++;
        if (_column >= columns()) {
            _column = 0;
            _row++;
            if (_row >= rows()) {
                _row = 0;
            }
        }
    }
    return value;
}

int TextLCD23017::_getc() {
    return -1;
}

void TextLCD23017::writeByte(int value) {//5 transactions of 3 bytes ~150bits ~1.5ms
    _intf.direction(PORT_A, PORT_DIR_OUT); //set to output
    _intf.write(PORT_A, value);//put value on data bus
    _intf.write(PORT_B, E|KEYS);//write instruction register, E=1
    _intf.write(PORT_B, KEYS);//E returns to 0
    _intf.direction(PORT_A, PORT_DIR_IN); //set to input
}

void TextLCD23017::writeCommand(int command) {//3 transactions 3 bytes ~0.9ms + busy + writeByte = 3.4ms
//    while (busy()) /*wait*/ ;
    writeByte(command);
    _intf.write(PORT_B, 0);//enable keys
}

void TextLCD23017::writeData(int data) {
//    while (busy()) /*wait*/ ;
    _intf.direction(PORT_A, PORT_DIR_OUT); //set to output
    _intf.write(PORT_A, data);//put data on data bus
    _intf.write(PORT_B, RS|KEYS);//setup RS
    _intf.write(PORT_B, E|RS|KEYS);//write data register, E=1, RS=1
    _intf.write(PORT_B, RS|KEYS);//E returns to 0
    _intf.direction(PORT_A, PORT_DIR_IN); //set to input
    _intf.write(PORT_B, 0);//enable keys
}

void TextLCD23017::writeString(char *s) {//convert a string into LCD commands
    char buffer[240];//size must be > 4*strlen(s) + 7
    char *p = strtok(s,"\n");
    _intf.direction(PORT_A, PORT_DIR_OUT); //set to output
    //assume RS=0, E=0, R/W = 0
    while (p) {
//    ::printf("section:'%s'\n", p);
        int i = 0;
        if (p != s) {//don't do this the first time (not a newline)
            _column = 0;
            if (++_row >= rows()) _row = 0;
        }
        buffer[i++] = E|KEYS;//B, clock RS and RW
        buffer[i++] = address(_column, _row);//A, this data is used
        buffer[i++] = KEYS;//B, clock data
        buffer[i++] = address(_column, _row);//A, dummy
        buffer[i++] = RS|KEYS;//B, setup RS
        for (int j = 0; j < strlen(p) && i<sizeof(buffer)-4; j++) {
            buffer[i++] = p[j];//A, dummy
            buffer[i++] = E|RS|KEYS;//B, clock RS and RW
            buffer[i++] = p[j];//A, this data is used
            buffer[i++] = RS|KEYS;//B, clock data
            if (++_column >= columns()) {
                _column = 0;
                if (++_row >= rows())  _row = 0;
            }
        }//RS=1, E=0, RW=0
        p = strtok(0, "\n");
        buffer[i++] = 0;//A, dummy
        buffer[i++] = (p!=0) ? KEYS : 0; //B, release RS
        _intf.write(PORT_B, buffer, i, p != 0);//start with B, send all substrings as a single transaction
    }
    _intf.direction(PORT_A, PORT_DIR_IN); //set to input
}

int TextLCD23017::address(int column, int row) {
    switch (_type) {
        case LCD20x4:
            switch (row) {
                case 0:
                    return 0x80 + column;
                case 1:
                    return 0xc0 + column;
                case 2:
                    return 0x94 + column;
                case 3:
                    return 0xd4 + column;
            }
        case LCD16x2B:
            return 0x80 + (row * 40) + column;
        case LCD16x2:
        case LCD20x2:
        default:
            return 0x80 + (row * 0x40) + column;
    }
}

int TextLCD23017::columns() {
    switch (_type) {
        case LCD20x4:
        case LCD20x2:
            return 20;
        case LCD16x2:
        case LCD16x2B:
        default:
            return 16;
    }
}

int TextLCD23017::rows() {
    switch (_type) {
        case LCD20x4:
            return 4;
        case LCD16x2:
        case LCD16x2B:
        case LCD20x2:
        default:
            return 2;
    }
}
